#include "oled_NHD-0216KZW-AB5.h"
#include "defines.h"
#include <delays.h>
#include <string.h>
#include <stdio.h>

static OLED_CHAR_DATA oled_char_data;
static OLED_CHAR_DATA *oled_char_data_p = &oled_char_data;

void NHD_Init() {
    PARALLEL_RS_TRIS = 0;
    PARALLEL_RW_TRIS = 0;
    PARALLEL_EN_TRIS = 0;

    PARALLEL_D4_TRIS = 0;
    PARALLEL_D5_TRIS = 0;
    PARALLEL_D6_TRIS = 0;
    PARALLEL_D7_TRIS = 0;

    oled_char_data_p->display_function = LCD_FUNCTIONSET | LCD_4BITMODE;
}

void NHD_Begin(char cols, char rows) {
    oled_char_data_p->num_lines = rows;
    oled_char_data_p->current_line = 0;

    PARALLEL_RS_LAT = 0;
    PARALLEL_RW_LAT = 0;
    PARALLEL_EN_LAT = 0;

    PARALLEL_D4_LAT = 0;
    PARALLEL_D5_LAT = 0;
    PARALLEL_D6_LAT = 0;
    PARALLEL_D7_LAT = 0;
    
    Delay10KTCYx(1);    // ~1ms

    // Initialization sequence
    NHD_Write_4_Bits(0x3);
    NHD_Write_4_Bits(0x2);
    NHD_Write_4_Bits(0x2);
    NHD_Write_4_Bits(0x8);
    NHD_Wait_For_Ready();

    NHD_Send_Command(0x08);  // Turn Off
    NHD_Send_Command(0x01);  // Clear Display
    NHD_Send_Command(0x06);  // Set Entry Mode
    NHD_Send_Command(0x02);  // Return to Home Position
    NHD_Send_Command(0x0C);  // Turn On
}

void NHD_Clear() {
    NHD_Send_Command(LCD_CLEARDISPLAY);
}

void NHD_Home() {
    NHD_Send_Command(LCD_RETURNHOME);
}

void NHD_Set_Cursor(unsigned char col, unsigned char row) {
    unsigned char row_offsets[] = {0x00, 0x40, 0x14, 0x54};
    if (row >= oled_char_data_p->num_lines) {
        row = 0;
    }
    NHD_Send_Command(LCD_SETDDRAMADDR | (col + row_offsets[row]));
}

void NHD_Display(char option) {
    if (option) {
        oled_char_data_p->display_control |= LCD_DISPLAYON;
    } else {
        oled_char_data_p->display_control &= ~LCD_DISPLAYON;
    }
    NHD_Send_Command(LCD_DISPLAYCONTROL | oled_char_data_p->display_control);
}

void NHD_Blink(char option) {
    if (option) {
        oled_char_data_p->display_control |= LCD_BLINKON;
    } else {
        oled_char_data_p->display_control &= ~LCD_BLINKON;
    }
    NHD_Send_Command(LCD_DISPLAYCONTROL | oled_char_data_p->display_control);
}

void NHD_Cursor(char option) {
    if (option) {
        oled_char_data_p->display_control |= LCD_CURSORON;
    } else {
        oled_char_data_p->display_control &= ~LCD_CURSORON;
    }
    NHD_Send_Command(LCD_DISPLAYCONTROL | oled_char_data_p->display_control);
}

void NHD_Scroll_Display_Left() {
    NHD_Send_Command(LCD_CURSORSHIFT | LCD_DISPLAYMOVE | LCD_MOVELEFT);
}

void NHD_Scroll_Display_Right() {
    NHD_Send_Command(LCD_CURSORSHIFT | LCD_DISPLAYMOVE | LCD_MOVERIGHT);
}

void NHD_Left_To_Rigtht() {
    oled_char_data_p->display_mode |= LCD_ENTRYLEFT;
    NHD_Send_Command(LCD_ENTRYMODESET | oled_char_data_p->display_mode);
}

void NHD_Right_To_Left() {
    oled_char_data_p->display_mode &= ~LCD_ENTRYLEFT;
    NHD_Send_Command(LCD_ENTRYMODESET | oled_char_data_p->display_mode);
}

void NHD_Autoscroll(char option) {
    if (option) {
        oled_char_data_p->display_mode |= LCD_ENTRYSHIFTINCREMENT;
    } else {
        oled_char_data_p->display_mode &= ~LCD_ENTRYSHIFTINCREMENT;
    }
    NHD_Send_Command(LCD_ENTRYMODESET | oled_char_data_p->display_mode);
}

void NHD_Create_Char(unsigned char location, unsigned char *charmap) {
    char i;
    location &= 0x7;
    NHD_Send_Command(LCD_SETCGRAMADDR | (location << 3));
    for (i = 0; i < 8; i++) {
        NHD_Send_Data(charmap[i]);
    }
}

void NHD_Send_Command(unsigned char value) {
    PARALLEL_RS_LAT = 0;
    PARALLEL_RW_LAT = 0;
    NHD_Write_4_Bits(value>>4);
    NHD_Write_4_Bits(value);
    NHD_Wait_For_Ready();
}

void NHD_Send_Data(unsigned char value) {
    PARALLEL_RS_LAT = 1;
    PARALLEL_RW_LAT = 0;
    NHD_Write_4_Bits(value>>4);
    NHD_Write_4_Bits(value);
    NHD_Wait_For_Ready();
}

void NHD_Pulse_Enable(void) {
    PARALLEL_EN_LAT = 1;
    Nop();
    Nop();
    PARALLEL_EN_LAT = 0;
}

void NHD_Write_4_Bits(unsigned char value) {
    PARALLEL_D4_LAT = (value) & 0x01;
    PARALLEL_D5_LAT = (value>>1) & 0x01;
    PARALLEL_D6_LAT = (value>>2) & 0x01;
    PARALLEL_D7_LAT = (value>>3) & 0x01;
    NHD_Pulse_Enable();
}

void NHD_Wait_For_Ready() {
    char busy;
    PARALLEL_BUSY_TRIS = 1;
    PARALLEL_RS_LAT = 0;
    PARALLEL_RW_LAT = 1;
    do {
        NHD_Pulse_Enable();
        Nop();
        busy = PARALLEL_BUSY_PORT;
        NHD_Pulse_Enable();
    } while (busy);
    PARALLEL_BUSY_TRIS = 0;
    PARALLEL_RW_LAT = 0;
}

void NHD_Write_String(const rom char *fmt, ...) {
    unsigned char i, len;
    unsigned char buffer[NHD_STRING_BUFFER_SIZE];

    // Parse and create string
    va_list args;
    va_start(args, fmt);
    vsprintf((char *) buffer, fmt, args);
    va_end(args);
    len = strlen((char *) buffer);

    // Make sure string to insert fits in buffer, truncate if necessary
    if (len > NHD_STRING_BUFFER_SIZE)
        len = NHD_STRING_BUFFER_SIZE;

    // Print buffer to string
    for (i = 0; i < len; i++) {
        NHD_Send_Data(buffer[i]);
    }
}