#include <delays.h>
#include <math.h>
#include "defines.h"
#include "sensor_temp_BMP085.h"
#include "base_I2C.h"


static BMP085_DATA *bmp085_data_p;

void BMP_Init(BMP085_DATA *data) {
    bmp085_data_p = data;
}

void BMP_Begin(char mode) {
    if (mode > BMP085_ULTRAHIGHRES)
        mode = BMP085_ULTRAHIGHRES;
    bmp085_data_p->oversampling = mode;

    if (BMP_Read8(0xD0) != 0x55) {
        DBG_PRINT_BMP("Error contacting BMP085!\r\n");
        return;
    }

    bmp085_data_p->ac1 = BMP_Read16(BMP085_CAL_AC1);
    bmp085_data_p->ac2 = BMP_Read16(BMP085_CAL_AC2);
    bmp085_data_p->ac3 = BMP_Read16(BMP085_CAL_AC3);
    bmp085_data_p->ac4 = BMP_Read16(BMP085_CAL_AC4);
    bmp085_data_p->ac5 = BMP_Read16(BMP085_CAL_AC5);
    bmp085_data_p->ac6 = BMP_Read16(BMP085_CAL_AC6);

    bmp085_data_p->b1 = BMP_Read16(BMP085_CAL_B1);
    bmp085_data_p->b2 = BMP_Read16(BMP085_CAL_B2);

    bmp085_data_p->mb = BMP_Read16(BMP085_CAL_MB);
    bmp085_data_p->mc = BMP_Read16(BMP085_CAL_MC);
    bmp085_data_p->md = BMP_Read16(BMP085_CAL_MD);

    DBG_PRINT_BMP("AC1 = %d\r\n", bmp085_data_p->ac1);
    DBG_PRINT_BMP("AC2 = %d\r\n", bmp085_data_p->ac2);
    DBG_PRINT_BMP("AC3 = %d\r\n", bmp085_data_p->ac3);
    DBG_PRINT_BMP("AC4 = %u\r\n", bmp085_data_p->ac4);
    DBG_PRINT_BMP("AC5 = %u\r\n", bmp085_data_p->ac5);
    DBG_PRINT_BMP("AC6 = %u\r\n", bmp085_data_p->ac6);

    DBG_PRINT_BMP("B1 = %d\r\n", bmp085_data_p->b1);
    DBG_PRINT_BMP("B2 = %d\r\n", bmp085_data_p->b2);

    DBG_PRINT_BMP("MB = %d\r\n", bmp085_data_p->mb);
    DBG_PRINT_BMP("MC = %d\r\n", bmp085_data_p->mc);
    DBG_PRINT_BMP("MD = %d\r\n", bmp085_data_p->md);
}

unsigned int BMP_Read_Raw_Temperature() {
    unsigned int ret;

    BMP_Write8(BMP085_CONTROL, BMP085_READTEMPCMD);
    Delay10KTCYx(255);
    ret = BMP_Read16(BMP085_TEMPDATA);
    
    DBG_PRINT_BMP("Raw Temp: %d\r\n", ret);
    
    return ret;
}

unsigned long BMP_Read_Raw_Pressure() {
    unsigned long ret;

    BMP_Write8(BMP085_CONTROL, BMP085_READPRESSURECMD + (bmp085_data_p->oversampling << 6));

    if (bmp085_data_p->oversampling == BMP085_ULTRALOWPOWER)
        Delay10KTCYx(255);
    else if (bmp085_data_p->oversampling == BMP085_STANDARD)
        Delay10KTCYx(255);
    else if (bmp085_data_p->oversampling == BMP085_HIGHRES)
        Delay10KTCYx(255);
    else
        Delay10KTCYx(255);

    ret = BMP_Read16(BMP085_PRESSUREDATA);
    ret <<= 8;
    ret |= BMP_Read8(BMP085_PRESSUREDATA+2);
    ret >>= (8 - bmp085_data_p->oversampling);

    DBG_PRINT_BMP("Raw Pressure: %ld\r\n", ret);

    return ret;
}

long BMP_Read_Pressure() {
    long UT, UP, B3, B5, B6, X1, X2, X3, p;
    unsigned long B4, B7;

    UT = BMP_Read_Raw_Temperature();
    UP = BMP_Read_Raw_Pressure();

    // Temperature calculations
    X1 = ((UT - (long) bmp085_data_p->ac6) * (long) bmp085_data_p->ac5) >> 15;
    X2 = ((long) bmp085_data_p->mc << 11);
    X2 -= (X1 + bmp085_data_p->md);
    X2 /= 2; // round up
    X2 /= (X1 + bmp085_data_p->md);
    B5 = X1 + X2;

    // Pressure calcs
    B6 = B5 - 4000;
    X1 = ((long) bmp085_data_p->b2 * ((B6 * B6) >> 12)) >> 11;
    X2 = ((long) bmp085_data_p->ac2 * B6) >> 11;
    X3 = X1 + X2;
    B3 = ((((long) bmp085_data_p->ac1 * 4 + X3) << bmp085_data_p->oversampling) + 2) / 4;

    X1 = ((long) bmp085_data_p->ac3 * B6) >> 13;
    X2 = ((long) bmp085_data_p->b1 * ((B6 * B6) >> 12)) >> 16;
    X3 = ((X1 + X2) + 2) >> 2;
    B4 = ((unsigned long) bmp085_data_p->ac4 * (unsigned long) (X3 + 32768)) >> 15;
    B7 = ((unsigned long) UP - B3) * (unsigned long) (50000UL >> bmp085_data_p->oversampling);

    if (B7 < 0x80000000) {
        p = (B7 * 2) / B4;
    } else {
        p = (B7 / B4) * 2;
    }
    X1 = (p >> 8) * (p >> 8);
    X1 = (X1 * 3038) >> 16;
    X2 = (-7357 * p) >> 16;

    p = p + ((X1 + X2 + (long)3791)>>4);

    return p;
}

float BMP_Read_Temperature() {
    long UT, X1, X2, B5;
    float temp;

    UT = BMP_Read_Raw_Temperature();

    X1 = ((UT - (long) bmp085_data_p->ac6) * (long) bmp085_data_p->ac5) >> 15;
    X2 = ((long) bmp085_data_p->mc << 11) / (X1 + (long) bmp085_data_p->md);
    B5 = X1 + X2;
    temp = (B5 + 8) >> 4;
    temp /= 10;

    return temp;
}

float BMP_Read_Altitude(float seaLevelPressure) {
    float altitude;
    float pressure = BMP_Read_Pressure();
    altitude = 44330 * (1.0 - pow(pressure /seaLevelPressure,0.1903));
    
    return altitude;
}

char BMP_Read8(char a) {
    char buffer[6], result, length, ret = 0;
    
    I2C_Master_Restart(BMP085_I2CADDR, a, 1);
    do {
        result = I2C_Get_Status();
    } while (!result);
    length = I2C_Read_Buffer((char *)buffer);
    ret = buffer[0];

    return ret;
}

unsigned int BMP_Read16(char a) {
    char buffer[6], result, length;
    unsigned int ret;

    I2C_Master_Restart(BMP085_I2CADDR, a, 2);
    do {
        result = I2C_Get_Status();
    } while (!result);
    length = I2C_Read_Buffer((char *)buffer);
    ret = buffer[0];
    ret <<= 8;
    ret |= buffer[1];

    return ret;
}

void BMP_Write8(char a, char d) {
    char buffer[2], result;
    buffer[0] = a;
    buffer[1] = d;

    I2C_Master_Send(BMP085_I2CADDR, 2, buffer);
    do {
        result = I2C_Get_Status();
    } while (!result);
}