#ifndef XBEE_H
#define XBEE_H

#define XBEE_BUFFER_SIZE        227

// If API mode = 2 is enabled
#define XBEE_USE_ESCAPE_CHAR

#define XBEE_ESCAPE_VAL         0x20
#define XBEE_START_DELIMITER    0x7E
#define XBEE_ESCAPE_CHAR        0x7D
#define XBEE_XON                0x11
#define XBEE_XOFF               0x13

// Expected 'next' state
#define XBEE_STATE_READ_START               10
#define XBEE_STATE_READ_LENGTH_HIGH         11
#define XBEE_STATE_READ_LENGTH_LOW          12
#define XBEE_STATE_READ_FRAME_DATA          13
#define XBEE_STATE_READ_CHECKSUM            14

// Command Frame Type
#define XBEE_TX_AT_COMMAND                  0x08
#define XBEE_TX_AT_COMMAND_QUEUE            0x09
#define XBEE_RX_AT_COMMAND_RESPONSE         0x88

#define XBEE_TX_DATA_PACKET                 0x10
#define XBEE_RX_DATA_PACKET                 0x90
#define XBEE_RX_DATA_TX_STATUS              0x8B
#define XBEE_RX_IO_DATA_SAMPLE              0x92
#define XBEE_TX_EXPLICIT_COMMAND            0x11
#define XBEE_RX_EXPLICIT_COMMAND            0x91

#define XBEE_TX_REMOTE_AT_COMMAND           0x17
#define XBEE_RX_REMOTE_AT_COMMAND_RESPONSE  0x97

#define XBEE_TX_CREATE_SOURCE_ROUTE         0x21
#define XBEE_RX_ROUTE_RECORD                0xA1
#define XBEE_RX_NODE_IDENTIFICATION         0x95
#define XBEE_RX_FRAME_MODEM_STATUS          0x8A

typedef struct {
    union {
        unsigned long long_value;
        char char_value[4]; // Little Endian!!
    } UPPER_32;
    union {
        unsigned long long_value;
        char char_value[4]; // Little Endian!!
    } LOWER_32;
} XBEE_ADDRESS_64;

typedef struct {
    union {
        unsigned int int_value;
        char char_value[2]; // Little Endian!!
    } INT_16;
} XBEE_ADDRESS_16;

// Unique Frame Components
typedef struct {
    char frame_type;
    char frame_id;
    char command[2];
    char data[XBEE_BUFFER_SIZE];
} XBEE_TX_AT_COMMAND_FRAME;
#define XBEE_TX_AT_COMMAND_FRAME_SIZE 4

typedef struct {
    char frame_type;
    char frame_id;
    char command[2];
    char data[XBEE_BUFFER_SIZE];
} XBEE_TX_AT_COMMAND_QUEUE_FRAME;
#define XBEE_TX_AT_COMMAND_QUEUE_FRAME_SIZE 4

typedef struct {
    char frame_type;
    char frame_id;
    char command[2];
    char command_status;
    char data[XBEE_BUFFER_SIZE];
} XBEE_RX_AT_COMMAND_RESPONSE_FRAME;
#define XBEE_RX_AT_COMMAND_RESPONSE_FRAME_SIZE 5

typedef struct {
    char frame_type;
    char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    char broadcast_radius;
    char options;
    char data[XBEE_BUFFER_SIZE];
} XBEE_TX_DATA_PACKET_FRAME;
#define XBEE_TX_DATA_PACKET_FRAME_SIZE 14

typedef struct {
    char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    char recieve_options;
    char data[XBEE_BUFFER_SIZE];
} XBEE_RX_DATA_PACKET_FRAME;
#define XBEE_RX_DATA_PACKET_FRAME_SIZE 12

typedef struct {
    char frame_type;
    char frame_id;
    XBEE_ADDRESS_16 destination_16;
    char transmit_retry_count;
    char delivery_status;
    char discovery_status;
} XBEE_RX_DATA_TX_STATUS_FRAME;
#define XBEE_RX_DATA_TX_STATUS_FRAME_SIZE 7

typedef struct {
    char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    char recieve_options;
    char number_of_samples;
    char digital_ch_mask[2];
    char analog_ch_mask;
    char digital_samples[2];
    char analog_samples[8];
} XBEE_RX_IO_DATA_SAMPLE_FRAME;
#define XBEE_RX_IO_DATA_SAMPLE_FRAME_SIZE 26

typedef struct {
    char frame_type;
    char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    char source_endpoint;
    char destination_endpoint;
    char cluster_id[2];
    char profile_id[2];
    char broadcast_radius;
    char transmit_options;
    char data[XBEE_BUFFER_SIZE];
} XBEE_TX_EXPLICIT_COMMAND_FRAME;
#define XBEE_TX_EXPLICIT_COMMAND_FRAME_SIZE 20

typedef struct {
    char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    char source_endpoint;
    char destination_endpoint;
    char cluster_id[2];
    char profile_id[2];
    char recieve_options;
    char data[XBEE_BUFFER_SIZE];
} XBEE_RX_EXPLICIT_COMMAND_FRAME;
#define XBEE_RX_EXPLICIT_COMMAND_FRAME_SIZE 18

typedef struct {
    char frame_type;
    char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    char remote_options;
    char command[2];
    char data[XBEE_BUFFER_SIZE];
} XBEE_TX_REMOTE_AT_COMMAND_FRAME;
#define XBEE_TX_REMOTE_AT_COMMAND_FRAME_SIZE 15

typedef struct {
    char frame_type;
    char frame_id;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    char command[2];
    char command_status;
    char command_data[4];
} XBEE_RX_REMOTE_AT_COMMAND_FRAME;
#define XBEE_RX_REMOTE_AT_COMMAND_FRAME_SIZE 19

typedef struct {
    char frame_type;
    char frame_id;
    XBEE_ADDRESS_64 destination_64;
    XBEE_ADDRESS_16 destination_16;
    char route_options;
    char num_of_addresses;
    char addresses[XBEE_BUFFER_SIZE];
} XBEE_TX_CREATE_SOURCE_ROUTE_FRAME;
#define XBEE_TX_CREATE_SOURCE_ROUTE_FRAME_SIZE 14

typedef struct {
    char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    char recieve_options;
    char num_of_addresses;
    char addresses[XBEE_BUFFER_SIZE];
} XBEE_RX_ROUTE_RECORD_FRAME;
#define XBEE_RX_ROUTE_RECORD_FRAME_SIZE 13

typedef struct {
    char frame_type;
    XBEE_ADDRESS_64 source_64;
    XBEE_ADDRESS_16 source_16;
    char recieve_options;
    XBEE_ADDRESS_16 remote_16;
    XBEE_ADDRESS_64 remote_64;
    char NI_string[2];
    XBEE_ADDRESS_16 parent_16;
    char device_type;
    char source_event;
    char profile_id[2];
    char manufacturer_id[2];
} XBEE_RX_NODE_IDENTIFICATION_INDICATOR_FRAME;
#define XBEE_RX_NODE_IDENTIFICATION_INDICATOR_FRAME_SIZE 32

typedef struct {
    char frame_type;
    char status;
} XBEE_RX_MODEM_STATUS_FRAME;
#define XBEE_RX_MODEM_STATUS_FRAME_SIZE 2

// Common Frame Components
typedef struct {
    char start_delimiter;
    XBEE_ADDRESS_16 length;
    union {
        XBEE_TX_AT_COMMAND_FRAME            TX_AT_COMMAND;
        XBEE_TX_AT_COMMAND_QUEUE_FRAME      TX_AT_COMMAND_QUEUE;
        XBEE_RX_AT_COMMAND_RESPONSE_FRAME   RX_AT_COMMAND_RESPONSE;
        XBEE_TX_DATA_PACKET_FRAME           TX_DATA_PACKET;
        XBEE_RX_DATA_PACKET_FRAME           RX_DATA_PACKET;
        XBEE_RX_DATA_TX_STATUS_FRAME        RX_DATA_TX_STATUS;
        XBEE_RX_IO_DATA_SAMPLE_FRAME        RX_IO_DATA_SAMPLE;
        XBEE_TX_EXPLICIT_COMMAND_FRAME      TX_EXPLICIT_COMMAND;
        XBEE_RX_EXPLICIT_COMMAND_FRAME      RX_EXPLICIT_COMMAND;
        XBEE_TX_REMOTE_AT_COMMAND_FRAME     TX_REMOTE_AT_COMMAND;
        XBEE_RX_REMOTE_AT_COMMAND_FRAME     RX_REMOTE_AT_COMMAND;
        XBEE_TX_CREATE_SOURCE_ROUTE_FRAME   TX_CREATE_SOURCE_ROUTE;
        XBEE_RX_ROUTE_RECORD_FRAME          RX_ROUTE_RECORD;
        XBEE_RX_NODE_IDENTIFICATION_INDICATOR_FRAME     RX_NODE_IDENTIFICATION;
        XBEE_RX_MODEM_STATUS_FRAME          RX_MODEM_STATUS;
    } FRAME;
} XBEE_FRAME;

// Overall Data Structure
typedef struct {
    XBEE_FRAME rcv_frame;
    unsigned int dataind;
    char checksum_sum;
    char read_state;
    char frame_rdy;
    char escape_flag;
} XBEE_DATA;


void XBee_Init(XBEE_DATA *data);
void XBee_Serial_In(char);
void XBee_Process_Received_Frame(void);
void XBee_Process_Transmit_Frame(char *data, char length);

unsigned int XBee_Get_Received_Frame(char *frame);

void XBee_Set_RTS(char);
char XBee_Read_CTS(void);

void XBee_Convert_Endian_64(XBEE_ADDRESS_64 *src);
void XBee_Convert_Endian_16(XBEE_ADDRESS_16 *src);

#endif