#include <xc.h>
#include "main.h"
#include "I2C.h"

static I2C_DATA *data_ptr;

// Set up the data structures for the base_I2C.code
// Should be called once before any i2c routines are called
void I2C_Init(I2C_DATA *data) {
    data_ptr = data;

    I2C_CLK_TRIS = 1;
    I2C_DAT_TRIS = 1;

    data_ptr->buffer_in_len = 0;
    data_ptr->buffer_in_len_tmp = 0;
    data_ptr->buffer_in_read_ind = 0;
    data_ptr->buffer_in_write_ind = 0;

    data_ptr->buffer_out_ind = 0;
    data_ptr->buffer_out_len = 0;

    data_ptr->operating_mode = 0;
    data_ptr->operating_state = I2C_IDLE;
    data_ptr->return_status = 0;

    data_ptr->slave_in_last_byte = 0;
    data_ptr->slave_sending_data = 0;

    data_ptr->master_dest_addr = 0;
    data_ptr->master_status = I2C_MASTER_IDLE;

    // Enable I2C interrupt
    PIE1bits.SSP1IE = 1;
}

// Setup the PIC to operate as a master.
void I2C_Configure_Master(char speed) {
    data_ptr->operating_mode = I2C_MODE_MASTER;

    SSP1STAT = 0x0;
    SSP1CON1 = 0x0;
    SSP1CON2 = 0x0;
    SSP1CON3 = 0x0;
    SSP1CON1bits.SSPM = 0x8; // I2C Master Mode
    if (speed) {
        SSP1ADD = 0x4F;      // Operate at 100KHz (32MHz)
    } else {
        SSP1ADD = 0x13;     // Operate at 400KHz (32MHz)
    }
    SSP1STATbits.SMP = 1;   // Disable Slew Rate Control
    SSP1CON3bits.PCIE = 1;  // Stop condition interrupt enable
    SSP1CON3bits.SCIE = 1;  // Start condition interrupt enable
    SSP1CON1bits.SSPEN = 1; // Enable MSSP Module
}

// Sends length number of bytes in msg to specified address (no R/W bit)
void I2C_Master_Send(char address, char *msg, char length) {
    char i;
    if (length == 0)
        return;

    // Copy message to send into buffer and save length/address
    for (i = 0; i < length; i++) {
        data_ptr->buffer_in[i] = msg[i];
    }
    data_ptr->buffer_in_len = length;
    data_ptr->master_dest_addr = address;
    data_ptr->buffer_in_read_ind = 0;
    data_ptr->buffer_in_write_ind = 0;

    // Change status to 'next' operation
    data_ptr->operating_state = I2C_SEND_ADDR;
    data_ptr->master_status = I2C_MASTER_SEND;

    // Generate start condition
    SSP1CON2bits.SEN = 1;
}

// Reads length number of bytes from address (no R/W bit)
void I2C_Master_Recv(char address, char length) {
    if (length == 0)
        return;

    // Save length and address to get data from
    data_ptr->buffer_in_len = length;
    data_ptr->master_dest_addr = address;
    data_ptr->buffer_in_read_ind = 0;
    data_ptr->buffer_in_write_ind = 0;

    // Change status to 'next' operation
    data_ptr->operating_state = I2C_SEND_ADDR;
    data_ptr->master_status = I2C_MASTER_RECV;

    // Generate start condition
    SSP1CON2bits.SEN = 1;
}

// Writes msg to address then reads length number of bytes from address
void I2C_Master_Restart(char address, char msg, char length) {
    char c;
    if (length == 0) {
        c = msg;
        I2C_Master_Send(address, &c, 1);
        return;
    }

    // Save length and address to get data from
    data_ptr->buffer_in[0] = msg;
    data_ptr->buffer_in_len = length;
    data_ptr->master_dest_addr = address;
    data_ptr->buffer_in_read_ind = 0;
    data_ptr->buffer_in_write_ind = 0;

    // Change status to 'next' operation
    data_ptr->operating_state = I2C_SEND_ADDR;
    data_ptr->master_status = I2C_MASTER_RESTART;

    // Generate start condition
    SSP1CON2bits.SEN = 1;
}

// Setup the PIC to operate as a slave. The address must not include the R/W bit
void I2C_Configure_Slave(char addr) {
    data_ptr->operating_mode = I2C_MODE_SLAVE;

    SSP1ADD = addr << 1;     // Set the slave address

    SSP1STAT = 0x0;
    SSP1CON1 = 0x0;
    SSP1CON2 = 0x0;
    SSP1CON3 = 0x0;
    SSP1CON1bits.SSPM = 0xE; // Enable Slave 7-bit w/ start/stop interrupts
    SSP1STATbits.SMP = 1;    // Slew Off
    SSP1CON2bits.SEN = 1;    // Enable clock-stretching
    SSP1CON1bits.SSPEN = 1;  // Enable MSSP Module
}

void I2C_Interrupt_Handler() {
    // Call interrupt depending on which mode we are operating in
    if (data_ptr->operating_mode == I2C_MODE_MASTER) {
        I2C_Interrupt_Master();
    } else if (data_ptr->operating_mode == I2C_MODE_SLAVE) {
        I2C_Interrupt_Slave();
    }
}

// An internal subroutine used in the master version of the i2c_interrupt_handler
void I2C_Interrupt_Master() {
    // If we are in the middle of sending data
    if (data_ptr->master_status == I2C_MASTER_SEND) {
        switch (data_ptr->operating_state) {
            case I2C_IDLE:
                break;
            case I2C_SEND_ADDR:
                // Send the address with read bit set
                data_ptr->operating_state = I2C_CHECK_ACK_SEND;
                SSP1BUF = (data_ptr->master_dest_addr << 1) | 0x0;
                break;
            case I2C_CHECK_ACK_SEND:
                // Check if ACK is received or not
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, send next byte of data
                    if (data_ptr->buffer_in_read_ind < data_ptr->buffer_in_len) {
                        SSP1BUF = data_ptr->buffer_in[data_ptr->buffer_in_read_ind];
                        data_ptr->buffer_in_read_ind++;
                    } else {
                        // If no more data is to be sent, send stop bit
                        data_ptr->operating_state = I2C_IDLE;
                        SSP1CON2bits.PEN = 1;
                        data_ptr->master_status = I2C_MASTER_IDLE;
                        data_ptr->return_status = I2C_SEND_OK;
                    }
                } else {
                    // If a NACK is received, stop transmission and send error
                    data_ptr->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    data_ptr->master_status = I2C_MASTER_IDLE;
                    data_ptr->return_status = I2C_SEND_FAIL;
                }
                break;
        }
    // If we are in the middle of receiving data
    } else if (data_ptr->master_status == I2C_MASTER_RECV) {
        switch (data_ptr->operating_state) {
            case I2C_IDLE:
                break;
            case I2C_SEND_ADDR:
                // Send address with write bit set
                data_ptr->operating_state = I2C_CHECK_ACK_RECV;
                char tmp = (data_ptr->master_dest_addr << 1);
                tmp |= 0x1;
                SSP1BUF = tmp;
                break;
            case I2C_CHECK_ACK_RECV:
                // Check if ACK is received
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, set module to receive 1 byte of data
                    data_ptr->operating_state = I2C_RCV_DATA;
                    SSP1CON2bits.RCEN = 1;
                } else {
                    // If a NACK is received, stop transmission and send error
                    data_ptr->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    data_ptr->master_status = I2C_MASTER_IDLE;
                    data_ptr->return_status = I2C_RECV_FAIL;
                }
                break;
            case I2C_RCV_DATA:
                // On receive, save byte into buffer
                // TODO: Handle I2C buffer overflow
                data_ptr->buffer_in[data_ptr->buffer_in_write_ind] = SSP1BUF;
                data_ptr->buffer_in_write_ind++;
                if (data_ptr->buffer_in_write_ind < data_ptr->buffer_in_len) {
                    // If we still need to read, send an ACK to the slave
                    data_ptr->operating_state = I2C_REQ_DATA;
                    SSP1CON2bits.ACKDT = 0;  // ACK
                    SSP1CON2bits.ACKEN = 1;
                } else {
                    // If we are done reading, send an NACK to the slave
                    data_ptr->operating_state = I2C_SEND_STOP;
                    SSP1CON2bits.ACKDT = 1;  // NACK
                    SSP1CON2bits.ACKEN = 1;
                }
                break;
            case I2C_REQ_DATA:
                // Set module to receive one byte of data
                data_ptr->operating_state = I2C_RCV_DATA;
                SSP1CON2bits.RCEN = 1;
                break;
            case I2C_SEND_STOP:
                // Send the stop bit and copy message to send to Main()
                data_ptr->operating_state = I2C_IDLE;
                SSP1CON2bits.PEN = 1;
                data_ptr->master_status = I2C_MASTER_IDLE;
                data_ptr->return_status = I2C_RECV_OK;
                break;
        }
    } else if (data_ptr->master_status == I2C_MASTER_RESTART) {
        switch (data_ptr->operating_state) {
            case I2C_IDLE:
                break;
            case I2C_SEND_ADDR:
                // Send the address with write bit set
                data_ptr->operating_state = I2C_CHECK_ACK_SEND;
                SSP1BUF = (data_ptr->master_dest_addr << 1) | 0x0;
                break;
            case I2C_CHECK_ACK_SEND:
                // Check if ACK is received or not
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, send first byte of data
                    SSP1BUF = data_ptr->buffer_in[0];
                    data_ptr->operating_state = I2C_CHECK_ACK_RESTART;
                } else {
                    // If a NACK is received, stop transmission and send error
                    data_ptr->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    data_ptr->master_status = I2C_MASTER_IDLE;
                    data_ptr->return_status = I2C_SEND_FAIL;
                }
                break;
            case I2C_CHECK_ACK_RESTART:
                if (!SSP1CON2bits.ACKSTAT) {
                    SSP1CON2bits.RSEN = 1;
                    data_ptr->operating_state = I2C_SEND_ADDR_2;
                } else {
                    // If a NACK is received, stop transmission and send error
                    data_ptr->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    data_ptr->master_status = I2C_MASTER_IDLE;
                    data_ptr->return_status = I2C_SEND_FAIL;
                }
                break;
            case I2C_SEND_ADDR_2:
                // Send the address with read bit set
                data_ptr->operating_state = I2C_CHECK_ACK_RECV;
                char tmp = (data_ptr->master_dest_addr << 1);
                tmp |= 0x1;
                SSP1BUF = tmp;
                break;
            case I2C_CHECK_ACK_RECV:
                // Check if ACK is received
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, set module to receive 1 byte of data
                    data_ptr->operating_state = I2C_RCV_DATA;
                    SSP1CON2bits.RCEN = 1;
                } else {
                    // If a NACK is received, stop transmission and send error
                    data_ptr->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    data_ptr->master_status = I2C_MASTER_IDLE;
                    data_ptr->return_status = I2C_RECV_FAIL;
                }
                break;
            case I2C_RCV_DATA:
                // On receive, save byte into buffer
                // TODO: Handle I2C buffer overflow
                data_ptr->buffer_in[data_ptr->buffer_in_write_ind] = SSP1BUF;
                data_ptr->buffer_in_write_ind++;
                if (data_ptr->buffer_in_write_ind < data_ptr->buffer_in_len) {
                    // If we still need to read, send an ACK to the slave
                    data_ptr->operating_state = I2C_REQ_DATA;
                    SSP1CON2bits.ACKDT = 0;  // ACK
                    SSP1CON2bits.ACKEN = 1;
                } else {
                    // If we are done reading, send an NACK to the slave
                    data_ptr->operating_state = I2C_SEND_STOP;
                    SSP1CON2bits.ACKDT = 1;  // NACK
                    SSP1CON2bits.ACKEN = 1;
                }
                break;
            case I2C_REQ_DATA:
                // Set module to receive one byte of data
                data_ptr->operating_state = I2C_RCV_DATA;
                SSP1CON2bits.RCEN = 1;
                break;
            case I2C_SEND_STOP:
                // Send the stop bit and copy message to send to Main()
                data_ptr->operating_state = I2C_IDLE;
                SSP1CON2bits.PEN = 1;
                data_ptr->master_status = I2C_MASTER_IDLE;
                data_ptr->return_status = I2C_RECV_OK;
                break;
        }
    }
}

void I2C_Interrupt_Slave() {
    char received_data;
    char data_read_from_buffer = 0;
    char data_written_to_buffer = 0;
    char overrun_error = 0;

    // Clear SSPOV (overflow bit)
    if (SSP1CON1bits.SSPOV == 1) {
        SSP1CON1bits.SSPOV = 0;
        // We failed to read the buffer in time, so we know we
        //  can't properly receive this message, just put us in the
        //  a state where we are looking for a new message
        data_ptr->operating_state = I2C_IDLE;
        overrun_error = 1;
        data_ptr->return_status = I2C_ERR_OVERRUN;
    }

    // Read SPPxBUF if it is full
    if (SSP1STATbits.BF == 1) {
        received_data = SSP1BUF;
        data_read_from_buffer = 1;
    }

    if (!overrun_error) {
        switch (data_ptr->operating_state) {
            case I2C_IDLE:
            {
                // Ignore anything except a start
                if (SSP1STATbits.S == 1) {
                    data_ptr->buffer_in_len_tmp = 0;
                    data_ptr->operating_state = I2C_STARTED;
//                    if (data_read_from_buffer) {
//                        if (SSPSTATbits.D_A == 1) {
//                            DBG_PRINT_I2C("I2C Start: (ERROR) no address recieved\r\n");
//                            // This is bad because we got data and we wanted an address
//                            data_ptr->operating_state = I2C_IDLE;
//                            data_ptr->return_status = I2C_ERR_NOADDR;
//                        } else {
//                            // Determine if we are sending or receiving data
//                            if (SSPSTATbits.R_W == 1) {
//                                data_ptr->operating_state = I2C_SEND_DATA;
//                            } else {
//                                data_ptr->operating_state = I2C_RCV_DATA;
//                            }
//                        }
//                    } else {
//                        data_ptr->operating_state = I2C_STARTED;
//                    }
                }
                break;
            }
            case I2C_STARTED:
            {
                // In this case, we expect either an address or a stop bit
                if (SSP1STATbits.P == 1) {
                    // Return to idle mode
                    data_ptr->operating_state = I2C_IDLE;
                } else if (data_read_from_buffer) {
                    if (SSP1STATbits.D_nA == 0) {
                        // Address received
                        if (SSP1STATbits.R_nW == 0) {
                            // Slave write mode
                            data_ptr->operating_state = I2C_RCV_DATA;
                        } else {
                            // Slave read mode
                            data_ptr->operating_state = I2C_SEND_DATA;
                            // Process the first byte immediatly if sending data
                            goto send;
                        }
                    } else {
                        data_ptr->operating_state = I2C_IDLE;
                        data_ptr->return_status = I2C_ERR_NODATA;
                    }
                }
                break;
            }
            send:
            case I2C_SEND_DATA:
            {
                if (!data_ptr->slave_sending_data) {
                    // If we are not currently sending data, figure out what to reply with
                    if (I2C_Process_Send(data_ptr->slave_in_last_byte)) {
                        // Data exists to be returned, send first byte
                        SSP1BUF = data_ptr->buffer_out[0];
                        data_ptr->buffer_out_ind = 1;
                        data_ptr->slave_sending_data = 1;
                        data_written_to_buffer = 1;
                    } else {
                        // Unknown request
                        data_ptr->slave_sending_data = 0;
                        data_ptr->operating_state = I2C_IDLE;
                    }
                } else {
                    // Sending remaining data back to master
                    if (data_ptr->buffer_out_ind < data_ptr->buffer_out_len) {
                        SSP1BUF = data_ptr->buffer_out[data_ptr->buffer_out_ind];
                        data_ptr->buffer_out_ind++;
                        data_written_to_buffer = 1;
                    } else {
                        // Nothing left to send
                        data_ptr->slave_sending_data = 0;
                        data_ptr->operating_state = I2C_IDLE;
                    }
                }
                break;
            }
            case I2C_RCV_DATA:
            {
                // We expect either data or a stop bit or a (if a restart, an addr)
                if (SSP1STATbits.P == 1) {
                    // Stop bit detected, we need to check to see if we also read data
                    if (data_read_from_buffer) {
                        if (SSP1STATbits.D_nA == 1) {
                            // Data received with stop bit
                            // TODO: Handle I2C buffer overflow
                            data_ptr->buffer_in[data_ptr->buffer_in_write_ind] = received_data;
                            if (data_ptr->buffer_in_write_ind == I2C_BUFFER_SIZE-1) {
                                data_ptr->buffer_in_write_ind = 0;
                            } else {
                                data_ptr->buffer_in_write_ind++;
                            }
                            data_ptr->buffer_in_len_tmp++;
                            // Save the last byte received
                            data_ptr->slave_in_last_byte = received_data;
                            data_ptr->return_status = I2C_DATA_AVAL;
                        } else {
                            data_ptr->operating_state = I2C_IDLE;
                            data_ptr->return_status = I2C_ERR_NODATA;
                        }
                    }
                    data_ptr->buffer_in_len += data_ptr->buffer_in_len_tmp;
                    data_ptr->operating_state = I2C_IDLE;
                } else if (data_read_from_buffer) {
                    if (SSP1STATbits.D_nA == 1) {
                        // Data received
                        data_ptr->buffer_in[data_ptr->buffer_in_write_ind] = received_data;
                        if (data_ptr->buffer_in_write_ind == I2C_BUFFER_SIZE-1) {
                            data_ptr->buffer_in_write_ind = 0;
                        } else {
                            data_ptr->buffer_in_write_ind++;
                        }
                        data_ptr->buffer_in_len_tmp++;
                        // Save the last byte received
                        data_ptr->slave_in_last_byte = received_data;
                        data_ptr->return_status = I2C_DATA_AVAL;
                    } else {
                        // Restart bit detected
                        if (SSP1STATbits.R_nW == 1) {
                            data_ptr->buffer_in_len += data_ptr->buffer_in_len_tmp;
                            data_ptr->operating_state = I2C_SEND_DATA;
                            // Process the first byte immediatly if sending data
                            goto send;
                        } else {
                            // Bad to recv an address again, we aren't ready
                            data_ptr->operating_state = I2C_IDLE;
                            data_ptr->return_status = I2C_ERR_NODATA;
                        }
                    }
                }
                break;
            }
        }
    }

    // Release the clock stretching bit (if we should)
    if (data_read_from_buffer || data_written_to_buffer) {
        // Release the clock
        if (SSP1CON1bits.CKP == 0) {
            SSP1CON1bits.CKP = 1;
        }
    }
}

/* Returns 0 if I2C module is currently busy, otherwise returns status code */
char I2C_Get_Status() {
    if (data_ptr->operating_mode == I2C_MODE_MASTER) {
        if (data_ptr->master_status != I2C_MASTER_IDLE || data_ptr->buffer_in_len == 0) {
            return 0;
        } else {
            return data_ptr->return_status;
        }
    } else {
        if (data_ptr->operating_state != I2C_IDLE || data_ptr->buffer_in_len == 0) {
            return 0;
        } else {
            return data_ptr->return_status;
        }
    }
}

char I2C_Buffer_Len() {
    return data_ptr->buffer_in_len;
}

/* Returns 0 if I2C module is currently busy, otherwise returns buffer length */
char I2C_Read_Buffer(char *buffer) {
    char i = 0;
    while (data_ptr->buffer_in_len != 0) {
        buffer[i] = data_ptr->buffer_in[data_ptr->buffer_in_read_ind];
        i++;
        if (data_ptr->buffer_in_read_ind == I2C_BUFFER_SIZE-1) {
            data_ptr->buffer_in_read_ind = 0;
        } else {
            data_ptr->buffer_in_read_ind++;
        }
        data_ptr->buffer_in_len--;
    }
    return i;
}

/* Put data to be returned here */
char I2C_Process_Send(char c) {
    char ret = 0;
    switch (c) {
        case 0xAA:
            data_ptr->buffer_out[0] = 'A';
            data_ptr->buffer_out_len = 1;
            ret = 1;
            break;
        case 0xBB:
            data_ptr->buffer_out[0] = '1';
            data_ptr->buffer_out[1] = '2';
            data_ptr->buffer_out_len = 2;
            ret = 1;
            break;
    }
    return ret;
}