// <editor-fold defaultstate="collapsed" desc="Configuration Bits">
// PIC16F1825 Configuration Bit Settings

// CONFIG1
#pragma config FOSC = INTOSC    // Oscillator Selection (INTOSC oscillator: I/O function on CLKIN pin)
#pragma config WDTE = OFF       // Watchdog Timer Enable (WDT enabled)
#pragma config PWRTE = OFF      // Power-up Timer Enable (PWRT disabled)
#pragma config MCLRE = OFF      // MCLR Pin Function Select (MCLR/VPP pin function is digital input)
#pragma config CP = OFF         // Flash Program Memory Code Protection (Program memory code protection is disabled)
#pragma config CPD = OFF        // Data Memory Code Protection (Data memory code protection is disabled)
#pragma config BOREN = ON       // Brown-out Reset Enable (Brown-out Reset enabled)
#pragma config CLKOUTEN = OFF   // Clock Out Enable (CLKOUT function is disabled. I/O or oscillator function on the CLKOUT pin)
#pragma config IESO = ON        // Internal/External Switchover (Internal/External Switchover mode is enabled)
#pragma config FCMEN = ON       // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is enabled)

// CONFIG2
#pragma config WRT = OFF        // Flash Memory Self-Write Protection (Write protection off)
#pragma config PLLEN = ON       // PLL Enable (4x PLL enabled)
#pragma config STVREN = ON      // Stack Overflow/Underflow Reset Enable (Stack Overflow or Underflow will cause a Reset)
#pragma config BORV = LO        // Brown-out Reset Voltage Selection (Brown-out Reset Voltage (Vbor), low trip point selected.)
#pragma config LVP = OFF        // Low-Voltage Programming Enable (High-voltage on MCLR/VPP must be used for programming)
// </editor-fold>

#include "defines.h"
#include "INTERRUPTS.h"
#include "STEPPER.h"
#include "IOC.h"
#include "SPI.h"
#include "ADC.h"
#include "OLED_SSD1306.h"
#include "TIMER.h"
#include <stdio.h>

void Pins_Init(void) {
    // RA0 and RA1 pins as analog input
    ANSELA = 0x3;
    ANSELC = 0x0;

//    // Enable weak pull-up if WPU bit is set
//    OPTION_REGbits.nWPUEN = 0;

    // SDO1 on RC2
    APFCON0bits.SDOSEL = 0;

    // Stepper Driver
    STEP_TRIS = 0;
    STEP_LAT = 0;
    M0_TRIS = 0;
    M0_LAT = 0;
    M1_TRIS = 0;
    M1_LAT = 0;
    M2_TRIS = 0;
    M2_LAT = 0;

    // I/O
    SW_1_TRIS = 1;
    SW_1_INLVL = 1;
    SW_2_TRIS = 1;
    SW_2_INLVL = 1;

    STEP_CURRENT_TRIS = 1;
    POT_CURRENT_TRIS = 1;

    // SPI
    SPI_MOSI_TRIS = 0;
    SPI_CLK_TRIS = 0;
    SPI_DC_SELECT_TRIS = 0;
    SPI_DC_SELECT_LAT = 0;
    SPI_RESET_TRIS = 0;
    SPI_RESET_LAT = 0;
}

OPERATING_MODE currMode;

int main(void) {
    // Set internal oscillator speed to 32MHz
    OSCCONbits.SPLLEN = 1;  // 4x PLL enable (overwritten by config bits)
    OSCCONbits.IRCF = 0xE;  // Base frequency @ 8MHz
    OSCCONbits.SCS = 0b00;  // System clock determined by config bits

    // Initialize I/O
    Pins_Init();

    IOC_Init();

    ADC_Init();

    TIMER_DATA timer_data;
    TIMER_Init(&timer_data);
    
    SPI_DATA spi_data;
    SPI_Init(&spi_data, SPI2_FOSC_16);

    SSD1306_DATA ssd1306_data;
    SSD1306_Init(&ssd1306_data);

    Interrupt_Init();
    Interrupt_Enable();

    currMode = SINGLE_STEP;

    SSD1306_Begin(SSD1306_SWITCHCAPVCC);

    while(1) {
        Update_OLED();
//        __delay_ms(1);
    }
}

void Set_Next_Mode() {
    switch (currMode) {
        case SINGLE_STEP:
            currMode = AUTO_STEP;
            break;
        case AUTO_STEP:
            currMode = SET_MICROSTEP;
            break;
        case SET_MICROSTEP:
        default:
            currMode = SINGLE_STEP;
            break;
    }
}

OPERATING_MODE Get_Cur_Mode(void) {
    return currMode;
}

void Update_OLED(void) {
    SSD1306_Clear_Display();
    SSD1306_Set_Text_Size(2);
    SSD1306_Set_Text_Wrap(0);
    switch (currMode) {
        case SINGLE_STEP:
            Draw_Manual_Text(1);
            Draw_Auto_Text(0);
            Draw_Step_Text(STEPPER_Get_Cur_Step(), 0);
            break;
        case AUTO_STEP:
            Draw_Manual_Text(0);
            Draw_Auto_Text(1);
            Draw_Step_Text(STEPPER_Get_Cur_Step(), 0);
            break;
        case SET_MICROSTEP:
            Draw_Manual_Text(0);
            Draw_Auto_Text(0);
            Draw_Step_Text(STEPPER_Get_Cur_Step(), 1);
            break;
    }
    Draw_Stepper_Current();
    Draw_Pot_Value();
    SSD1306_Display();
}

void Draw_Manual_Text(uint8_t selected) {
    // Draw and/or highlight the stepping mode in the top left corner
    uint8_t stringManual[] = "MANUAL";
    if (selected) {
        SSD1306_Fill_Rect(0, 0, 75, 16, 1);
        SSD1306_Set_Text_Color(0);
    } else {
        SSD1306_Set_Text_Color(1);
    }
    SSD1306_Set_Cursor(3, 1);
    SSD1306_Write_String(stringManual, 6);
}

void Draw_Auto_Text(uint8_t selected) {
    // Draw and/or highlight the stepping mode in the top right corner
    uint8_t stringAuto[] = "AUTO";
    if (selected) {
        SSD1306_Fill_Rect(76, 0, 53, 16, 1);
        SSD1306_Set_Text_Color(0);
    } else {
        SSD1306_Set_Text_Color(1);
    }
    SSD1306_Set_Cursor(79, 1);
    SSD1306_Write_String(stringAuto, 4);
}

void Draw_Stepper_Current() {
    // Draw the stepper motor current draw in the bottom right corner
    uint8_t buffer[6] = {0};
    uint16_t potVoltage = ADC_Read(STEP_ADC_CHANNEL);
    // Compute current from voltage reading (2x)
    uint16_t current = potVoltage * 0.3222 * 2;
    uint8_t preDecimal = current / 100;
    uint8_t postDecimal = current % 100;
    sprintf(buffer, "%1d.%02dA", preDecimal, postDecimal);
    SSD1306_Set_Text_Color(1);
    SSD1306_Set_Cursor(67, 17);
    SSD1306_Write_String(buffer, 5);
}

void Draw_Pot_Value() {
    // Draw a line indicating auto rotation delay
    uint8_t potVoltage = ADC_Read(POT_ADC_CHANNEL) >> 4;
    SSD1306_Draw_Line(127, 31, 127 - potVoltage, 31, 1);
}

void Draw_Step_Text(STEPPER_MICROSTEP step, uint8_t selected) {
    // Draw and/or highlight the stepping in the bottom left corner
    uint8_t stringStepping32[] = "1/32";
    uint8_t stringStepping16[] = "1/16";
    uint8_t stringStepping8[] = "1/8";
    uint8_t stringStepping4[] = "1/4";
    uint8_t stringStepping2[] = "1/2";
    uint8_t stringStepping1[] = "1/1";
    if (selected) {
        SSD1306_Fill_Rect(0, 16, 52, 16, 1);
        SSD1306_Set_Text_Color(0);
    } else {
        SSD1306_Set_Text_Color(1);
    }
    if (step == STEP_1_32) {
        SSD1306_Set_Cursor(2, 17);
        SSD1306_Write_String(stringStepping32, 4);
    } else if (step == STEP_1_16) {
        SSD1306_Set_Cursor(2, 17);
        SSD1306_Write_String(stringStepping16, 4);
    } else if (step == STEP_1_8) {
        SSD1306_Set_Cursor(8, 17);
        SSD1306_Write_String(stringStepping8, 3);
    } else if (step == STEP_1_4) {
        SSD1306_Set_Cursor(8, 17);
        SSD1306_Write_String(stringStepping4, 3);
    } else if (step == STEP_1_2) {
        SSD1306_Set_Cursor(8, 17);
        SSD1306_Write_String(stringStepping2, 3);
    } else if (step == STEP_1_1) {
        SSD1306_Set_Cursor(8, 17);
        SSD1306_Write_String(stringStepping1, 3);
    }
}