#include "defines.h"
#include "I2C1.h"

static I2C1_DATA *i2c_data_p;

// Set up the data structures for the base_I2C.code
// Should be called once before any i2c routines are called
void I2C1_Init(I2C1_DATA *data) {
    i2c_data_p = data;
    
    i2c_data_p->buffer_in_len = 0;
    i2c_data_p->buffer_in_len_tmp = 0;
    i2c_data_p->buffer_in_read_ind = 0;
    i2c_data_p->buffer_in_write_ind = 0;

    i2c_data_p->buffer_out_ind = 0;
    i2c_data_p->buffer_out_len = 0;
    
    i2c_data_p->operating_mode = 0;
    i2c_data_p->operating_state = I2C_IDLE;
    i2c_data_p->return_status = 0;

    i2c_data_p->slave_in_last_byte = 0;
    i2c_data_p->slave_sending_data = 0;

    i2c_data_p->master_dest_addr = 0;
    i2c_data_p->master_status = I2C_MASTER_IDLE;
    
    // Enable I2C interrupt
    PIE1bits.SSP1IE = 1;
}

// Setup the PIC to operate as a master.
void I2C1_Configure_Master(uint8_t speed) {
    i2c_data_p->operating_mode = I2C_MODE_MASTER;

    I2C_1_CLK_TRIS = 1;
    I2C_1_DAT_TRIS = 1;

    SSP1STAT = 0x0;
    SSP1CON1 = 0x0;
    SSP1CON2 = 0x0;
    SSP1CON3 = 0x0;
    SSP1CON1bits.SSPM = 0x8; // I2C Master Mode
    if (speed == 0x01) {
        SSP1ADD = 0x13;         // Operate at 400KHz (32MHz)
        SSP1STATbits.SMP = 1;    // Disable Slew Rate Control
    } else if (speed == 0x02) {
        SSP1ADD = 0x07;         // Operate at 1Mhz (32Mhz)
        SSP1STATbits.SMP = 1;    // Disable Slew Rate Control
    } else {
        SSP1ADD = 0x4F;         // Operate at 100KHz (32MHz)
        SSP1STATbits.SMP = 0;    // Enable Slew Rate Control
    }
    SSP1CON1bits.SSPEN = 1;  // Enable MSSP1 Module
}

// Sends length number of bytes in msg to specified address (no R/W bit)
void I2C1_Master_Send(uint8_t address, uint8_t *msg, uint8_t length) {
    uint8_t i;
    if (length == 0)
        return;
    
    // Copy message to send into buffer and save length/address
    for (i = 0; i < length; i++) {
        i2c_data_p->buffer_in[i] = msg[i];
    }
    i2c_data_p->buffer_in_len = length;
    i2c_data_p->master_dest_addr = address;
    i2c_data_p->buffer_in_read_ind = 0;
    i2c_data_p->buffer_in_write_ind = 0;

    // Change status to 'next' operation
    i2c_data_p->operating_state = I2C_SEND_ADDR;
    i2c_data_p->master_status = I2C_MASTER_SEND;
    
    // Generate start condition
    SSP1CON2bits.SEN = 1;
}

// Reads length number of bytes from address (no R/W bit)
void I2C1_Master_Recv(uint8_t address, uint8_t length) {
    if (length == 0)
        return;

    // Save length and address to get data from
    i2c_data_p->buffer_in_len = length;
    i2c_data_p->master_dest_addr = address;
    i2c_data_p->buffer_in_read_ind = 0;
    i2c_data_p->buffer_in_write_ind = 0;

    // Change status to 'next' operation
    i2c_data_p->operating_state = I2C_SEND_ADDR;
    i2c_data_p->master_status = I2C_MASTER_RECV;
    
    // Generate start condition
    SSP1CON2bits.SEN = 1;
}

// Writes msg to address then reads length number of bytes from address
void I2C1_Master_Restart(uint8_t address, uint8_t msg, uint8_t length) {
    uint8_t c;
    if (length == 0) {
        c = msg;
        I2C1_Master_Send(address, &c, 1);
        return;
    }

    // Save length and address to get data from
    i2c_data_p->buffer_in[0] = msg;
    i2c_data_p->buffer_in_len = length;
    i2c_data_p->master_dest_addr = address;
    i2c_data_p->buffer_in_read_ind = 0;
    i2c_data_p->buffer_in_write_ind = 0;

    // Change status to 'next' operation
    i2c_data_p->operating_state = I2C_SEND_ADDR;
    i2c_data_p->master_status = I2C_MASTER_RESTART;

    // Generate start condition
    SSP1CON2bits.SEN = 1;
}

// Setup the PIC to operate as a slave. The address must not include the R/W bit
void I2C1_Configure_Slave(uint8_t addr) {
    i2c_data_p->operating_mode = I2C_MODE_SLAVE;

    // Ensure the two lines are set for input (we are a slave)
    I2C_1_CLK_TRIS = 1;
    I2C_1_DAT_TRIS = 1;

    SSP1ADD = addr << 1;     // Set the slave address

    SSP1STAT = 0x0;
    SSP1CON1 = 0x0;
    SSP1CON2 = 0x0;
    SSP1CON3 = 0x0;
    SSP1CON1bits.SSPM = 0x6; // Enable Slave 7-bit address
    SSP1STATbits.SMP = 1;    // Slew Off
    SSP1CON2bits.SEN = 1;    // Enable clock-stretching
    SSP1CON3bits.PCIE = 1;   // Interrupt on stop condition
    SSP1CON3bits.SCIE = 0;   // Disable interrupt on start condition
    SSP1CON1bits.SSPEN = 1;  // Enable MSSP1 Module
}

void I2C1_Interrupt_Handler() {
    // Call interrupt depending on which mode we are operating in
    if (i2c_data_p->operating_mode == I2C_MODE_MASTER) {
        I2C1_Interrupt_Master();
    } else if (i2c_data_p->operating_mode == I2C_MODE_SLAVE) {
        I2C1_Interrupt_Slave();
    }
}

// An internal subroutine used in the master version of the i2c_interrupt_handler
void I2C1_Interrupt_Master() {
    // If we are in the middle of sending data
    if (i2c_data_p->master_status == I2C_MASTER_SEND) {
        switch (i2c_data_p->operating_state) {
            case I2C_IDLE:
                break;
            case I2C_SEND_ADDR:
                // Send the address with read bit set
                i2c_data_p->operating_state = I2C_CHECK_ACK_SEND;
                SSP1BUF = (i2c_data_p->master_dest_addr << 1) | 0x0;
                break;
            case I2C_CHECK_ACK_SEND:
                // Check if ACK is received or not
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, send next byte of data
                    if (i2c_data_p->buffer_in_read_ind < i2c_data_p->buffer_in_len) {
                        SSP1BUF = i2c_data_p->buffer_in[i2c_data_p->buffer_in_read_ind];
                        i2c_data_p->buffer_in_read_ind++;
                    } else {
                        // If no more data is to be sent, send stop bit
                        i2c_data_p->operating_state = I2C_IDLE;
                        SSP1CON2bits.PEN = 1;
                        i2c_data_p->master_status = I2C_MASTER_IDLE;
                        i2c_data_p->return_status = I2C_SEND_OK;
                    }
                } else {
                    // If a NACK is received, stop transmission and send error
                    i2c_data_p->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    i2c_data_p->master_status = I2C_MASTER_IDLE;
                    i2c_data_p->return_status = I2C_SEND_FAIL;
                }
                break;
        }
    // If we are in the middle of receiving data
    } else if (i2c_data_p->master_status == I2C_MASTER_RECV) {
        switch (i2c_data_p->operating_state) {
            case I2C_IDLE:
                break;
            case I2C_SEND_ADDR:
                // Send address with write bit set
                i2c_data_p->operating_state = I2C_CHECK_ACK_RECV;
                uint8_t tmp = (i2c_data_p->master_dest_addr << 1);
                tmp |= 0x01;
                SSP1BUF = tmp;
                break;
            case I2C_CHECK_ACK_RECV:
                // Check if ACK is received
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, set module to receive 1 byte of data
                    i2c_data_p->operating_state = I2C_RCV_DATA;
                    SSP1CON2bits.RCEN = 1;
                } else {
                    // If a NACK is received, stop transmission and send error
                    i2c_data_p->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    i2c_data_p->master_status = I2C_MASTER_IDLE;
                    i2c_data_p->return_status = I2C_RECV_FAIL;
                }
                break;
            case I2C_RCV_DATA:
                // On receive, save byte into buffer
                // TODO: Handle I2C buffer overflow
                i2c_data_p->buffer_in[i2c_data_p->buffer_in_write_ind] = SSP1BUF;
                i2c_data_p->buffer_in_write_ind++;
                if (i2c_data_p->buffer_in_write_ind < i2c_data_p->buffer_in_len) {
                    // If we still need to read, send an ACK to the slave
                    i2c_data_p->operating_state = I2C_REQ_DATA;
                    SSP1CON2bits.ACKDT = 0;  // ACK
                    SSP1CON2bits.ACKEN = 1;
                } else {
                    // If we are done reading, send an NACK to the slave
                    i2c_data_p->operating_state = I2C_SEND_STOP;
                    SSP1CON2bits.ACKDT = 1;  // NACK
                    SSP1CON2bits.ACKEN = 1;
                }
                break;
            case I2C_REQ_DATA:
                // Set module to receive one byte of data
                i2c_data_p->operating_state = I2C_RCV_DATA;
                SSP1CON2bits.RCEN = 1;
                break;
            case I2C_SEND_STOP:
                // Send the stop bit and copy message to send to Main()
                i2c_data_p->operating_state = I2C_IDLE;
                SSP1CON2bits.PEN = 1;
                i2c_data_p->master_status = I2C_MASTER_IDLE;
                i2c_data_p->return_status = I2C_RECV_OK;
                break;
        }
    } else if (i2c_data_p->master_status == I2C_MASTER_RESTART) {
        switch (i2c_data_p->operating_state) {
            case I2C_IDLE:
                break;
            case I2C_SEND_ADDR:
                // Send the address with read bit set
                i2c_data_p->operating_state = I2C_CHECK_ACK_SEND;
                SSP1BUF = (i2c_data_p->master_dest_addr << 1) | 0x0;
                break;
            case I2C_CHECK_ACK_SEND:
                // Check if ACK is received or not
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, send first byte of data
                    SSP1BUF = i2c_data_p->buffer_in[0];
                    i2c_data_p->operating_state = I2C_CHECK_ACK_RESTART;
                } else {
                    // If a NACK is received, stop transmission and send error
                    i2c_data_p->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    i2c_data_p->master_status = I2C_MASTER_IDLE;
                    i2c_data_p->return_status = I2C_SEND_FAIL;
                }
                break;
            case I2C_CHECK_ACK_RESTART:
                if (!SSP1CON2bits.ACKSTAT) {
                    SSP1CON2bits.RSEN = 1;
                    i2c_data_p->operating_state = I2C_SEND_ADDR_2;
                } else {
                    // If a NACK is received, stop transmission and send error
                    i2c_data_p->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    i2c_data_p->master_status = I2C_MASTER_IDLE;
                    i2c_data_p->return_status = I2C_SEND_FAIL;
                }
                break;
            case I2C_SEND_ADDR_2:
                // Send the address with read bit set
                i2c_data_p->operating_state = I2C_CHECK_ACK_RECV;
                uint8_t tmp = (i2c_data_p->master_dest_addr << 1);
                tmp |= 0x01;
                SSP1BUF = tmp;
                break;
            case I2C_CHECK_ACK_RECV:
                // Check if ACK is received
                if (!SSP1CON2bits.ACKSTAT) {
                    // If an ACK is received, set module to receive 1 byte of data
                    i2c_data_p->operating_state = I2C_RCV_DATA;
                    SSP1CON2bits.RCEN = 1;
                } else {
                    // If a NACK is received, stop transmission and send error
                    i2c_data_p->operating_state = I2C_IDLE;
                    SSP1CON2bits.PEN = 1;
                    i2c_data_p->master_status = I2C_MASTER_IDLE;
                    i2c_data_p->return_status = I2C_RECV_FAIL;
                }
                break;
            case I2C_RCV_DATA:
                // On receive, save byte into buffer
                // TODO: Handle I2C buffer overflow
                i2c_data_p->buffer_in[i2c_data_p->buffer_in_write_ind] = SSP1BUF;
                i2c_data_p->buffer_in_write_ind++;
                if (i2c_data_p->buffer_in_write_ind < i2c_data_p->buffer_in_len) {
                    // If we still need to read, send an ACK to the slave
                    i2c_data_p->operating_state = I2C_REQ_DATA;
                    SSP1CON2bits.ACKDT = 0;  // ACK
                    SSP1CON2bits.ACKEN = 1;
                } else {
                    // If we are done reading, send an NACK to the slave
                    i2c_data_p->operating_state = I2C_SEND_STOP;
                    SSP1CON2bits.ACKDT = 1;  // NACK
                    SSP1CON2bits.ACKEN = 1;
                }
                break;
            case I2C_REQ_DATA:
                // Set module to receive one byte of data
                i2c_data_p->operating_state = I2C_RCV_DATA;
                SSP1CON2bits.RCEN = 1;
                break;
            case I2C_SEND_STOP:
                // Send the stop bit
                i2c_data_p->operating_state = I2C_IDLE;
                SSP1CON2bits.PEN = 1;
                i2c_data_p->master_status = I2C_MASTER_IDLE;
                i2c_data_p->return_status = I2C_RECV_OK;
                break;
        }
    }
}

void I2C1_Interrupt_Slave() {
    uint8_t received_data;
    uint8_t data_read_from_buffer = 0;
    uint8_t data_written_to_buffer = 0;
    uint8_t overrun_error = 0;

    // Clear SSPOV (overflow bit)
    if (SSP1CON1bits.SSPOV == 1) {
        SSP1CON1bits.SSPOV = 0;
        // We failed to read the buffer in time, so we know we
        //  can't properly receive this message, just put us in the
        //  a state where we are looking for a new message
        i2c_data_p->operating_state = I2C_IDLE;
        overrun_error = 1;
        i2c_data_p->return_status = I2C_ERR_OVERRUN;
    }

    // Read SPPxBUF if it is full
    if (SSP1STATbits.BF == 1) {
        received_data = SSP1BUF;
        data_read_from_buffer = 1;
    }

    if (!overrun_error) {
        switch (i2c_data_p->operating_state) {
            case I2C_IDLE:
            {
//                // Ignore anything except a start
//                if (SSP1STATbits.S == 1) {
//                    i2c_data_p->buffer_in_len_tmp = 0;
//                    i2c_data_p->operating_state = I2C_STARTED;
//                }
//                break;
//            }
//            case I2C_STARTED:
//            {
                // In this case, we expect either an address or a stop bit
                if (SSP1STATbits.P == 1) {
                    // Return to idle mode
                    i2c_data_p->operating_state = I2C_IDLE;
                } else if (data_read_from_buffer) {
                    i2c_data_p->buffer_in_len_tmp = 0;
                    if (SSP1STATbits.D_nA == 0) {
                        // Address received
                        if (SSP1STATbits.R_nW == 0) {
                            // Slave write mode
                            i2c_data_p->operating_state = I2C_RCV_DATA;
                        } else {
                            // Slave read mode
                            i2c_data_p->operating_state = I2C_SEND_DATA;
                            // Process the first byte immediatly if sending data
                            goto send;
                        }
                    } else {
                        i2c_data_p->operating_state = I2C_IDLE;
                        i2c_data_p->return_status = I2C_ERR_NOADDR;
                    }
                }
                break;
            }
            send:
            case I2C_SEND_DATA:
            {
                if (!i2c_data_p->slave_sending_data) {
                    // If we are not currently sending data, figure out what to reply with
                    if (I2C1_Process_Receive(i2c_data_p->slave_in_last_byte)) {
                        // Data exists to be returned, send first byte
                        SSP1BUF = i2c_data_p->buffer_out[0];
                        i2c_data_p->buffer_out_ind = 1;
                        i2c_data_p->slave_sending_data = 1;
                        data_written_to_buffer = 1;
                    } else {
                        // Unknown request
                        i2c_data_p->slave_sending_data = 0;
                        i2c_data_p->operating_state = I2C_IDLE;
                    }
                } else {
                    // Sending remaining data back to master
                    if (i2c_data_p->buffer_out_ind < i2c_data_p->buffer_out_len) {
                        SSP1BUF = i2c_data_p->buffer_out[i2c_data_p->buffer_out_ind];
                        i2c_data_p->buffer_out_ind++;
                        data_written_to_buffer = 1;
                    } else {
                        // Nothing left to send
                        i2c_data_p->slave_sending_data = 0;
                        i2c_data_p->operating_state = I2C_IDLE;
                    }
                }
                break;
            }
            case I2C_RCV_DATA:
            {
                // We expect either data or a stop bit or a (if a restart, an addr)
                if (SSP1STATbits.P == 1) {
                    // Stop bit detected, we need to check to see if we also read data
                    if (data_read_from_buffer) {
                        if (SSP1STATbits.D_nA == 1) {
                            // Data received with stop bit
                            // TODO: Handle I2C buffer overflow
                            i2c_data_p->buffer_in[i2c_data_p->buffer_in_write_ind] = received_data;
                            if (i2c_data_p->buffer_in_write_ind == MAXI2C1BUF-1) {
                                i2c_data_p->buffer_in_write_ind = 0;
                            } else {
                                i2c_data_p->buffer_in_write_ind++;
                            }
                            i2c_data_p->buffer_in_len_tmp++;
                            // Save the last byte received
                            i2c_data_p->slave_in_last_byte = received_data;
                            i2c_data_p->return_status = I2C_DATA_AVAL;
                        } else {
                            i2c_data_p->operating_state = I2C_IDLE;
                            i2c_data_p->return_status = I2C_ERR_NODATA;
                        }
                    }
                    i2c_data_p->buffer_in_len += i2c_data_p->buffer_in_len_tmp;
                    i2c_data_p->operating_state = I2C_IDLE;
                } else if (data_read_from_buffer) {
                    if (SSP1STATbits.D_nA == 1) {
                        // Data received
                        i2c_data_p->buffer_in[i2c_data_p->buffer_in_write_ind] = received_data;
                        if (i2c_data_p->buffer_in_write_ind == MAXI2C1BUF-1) {
                            i2c_data_p->buffer_in_write_ind = 0;
                        } else {
                            i2c_data_p->buffer_in_write_ind++;
                        }
                        i2c_data_p->buffer_in_len_tmp++;
                        // Save the last byte received
                        i2c_data_p->slave_in_last_byte = received_data;
                        i2c_data_p->return_status = I2C_DATA_AVAL;
                    } else {
                        // Restart bit detected
                        if (SSP1STATbits.R_nW == 1) {
                            i2c_data_p->buffer_in_len += i2c_data_p->buffer_in_len_tmp;
                            i2c_data_p->operating_state = I2C_SEND_DATA;
                            // Process the first byte immediatly if sending data
                            goto send;
                        } else {
                            // Bad to recv an address again, we aren't ready
                            i2c_data_p->operating_state = I2C_IDLE;
                            i2c_data_p->return_status = I2C_ERR_NODATA;
                        }
                    }
                }
                break;
            }
        }
    }

    // Release the clock stretching bit (if we should)
    if (data_read_from_buffer || data_written_to_buffer) {
        // Release the clock
        if (SSP1CON1bits.CKP == 0) {
            SSP1CON1bits.CKP = 1;
        }
    }
}

/* Returns 0 if I2C module is currently busy, otherwise returns status code */
uint8_t I2C1_Get_Status() {
    if (i2c_data_p->operating_mode == I2C_MODE_MASTER) {
        if (i2c_data_p->master_status != I2C_MASTER_IDLE || i2c_data_p->buffer_in_len == 0) {
            return 0;
        } else {
            return i2c_data_p->return_status;
        }
    } else {
        if (i2c_data_p->operating_state != I2C_IDLE || i2c_data_p->buffer_in_len == 0) {
            return 0;
        } else {
            return i2c_data_p->return_status;
        }
    }
}

uint8_t I2C1_Buffer_Len() {
    return i2c_data_p->buffer_in_len;
}

/* Returns 0 if I2C module is currently busy, otherwise returns buffer length */
uint8_t I2C1_Read_Buffer(uint8_t *buffer) {
    uint8_t i = 0;
    while (i2c_data_p->buffer_in_len != 0) {
        buffer[i] = i2c_data_p->buffer_in[i2c_data_p->buffer_in_read_ind];
        i++;
        if (i2c_data_p->buffer_in_read_ind == MAXI2C1BUF-1) {
            i2c_data_p->buffer_in_read_ind = 0;
        } else {
            i2c_data_p->buffer_in_read_ind++;
        }
        i2c_data_p->buffer_in_len--;
    }
    return i;
}

/* Put data to be returned here */
uint8_t I2C1_Process_Receive(uint8_t c) {
    uint8_t ret = 0;

    return ret;
}