#ifndef CUBE_H
#define	CUBE_H

// Cube Parameters
#define CUBE_ROW_COUNT 8
#define CUBE_COLUMN_COUNT 8
#define CUBE_LAYER_COUNT 8
#define CUBE_PIXELS 512
#define CUBE_ROTATIONS 7

#define GCS_REG_SIZE 36
#define GCS_LAYER_SIZE (GCS_REG_SIZE*CUBE_ROW_COUNT)

// Color Definitions
#define CLEAR  0x000,0x000,0x000
#define RED    0x0FF,0x000,0x000
#define ORANGE 0x0FF,0x020,0x000
#define YELLOW 0x0FF,0x060,0x000
#define GREEN  0x000,0x0FF,0x000
#define TEAL   0x000,0x0FF,0x040
#define BLUE   0x000,0x000,0x0FF
#define PURPLE 0x0FF,0x000,0x0FF
#define WHITE  0x0FF,0x0FF,0x0FF

// Control Pin Declarations
#define DCSIN_TRIS TRISDbits.TRISD3
#define DCSCK_TRIS TRISDbits.TRISD12

#define DCSIN LATDbits.LATD3
#define DCSCK LATDbits.LATD12

#define SFT_R_TRIS TRISBbits.TRISB15
#define SFT_K_TRIS TRISDbits.TRISD5
#define SFT_S_TRIS TRISDbits.TRISD4
#define SFT_D_TRIS TRISBbits.TRISB14

#define SFT_R LATBbits.LATB15
#define SFT_K LATDbits.LATD5
#define SFT_S LATDbits.LATD4
#define SFT_D LATBbits.LATB14

#define GSLAT_TRIS TRISDbits.TRISD9
#define XBLNK_TRIS TRISDbits.TRISD2

#define GSLAT LATDbits.LATD9
#define XBLNK LATDbits.LATD2

// String Overlay Buffer Size
#define CUBE_STRING_MAX_LENGTH 255

// Data Streaming In Buffer Size
#define CUBE_FRAME_BUFFER_SIZE 3000
#define CUBE_START_CHAR  0x7E
#define CUBE_ESCAPE_CHAR 0x7D
#define CUBE_ESCAPE_XOR  0x20

// Data Streaming In Command Set
#define CUBE_COMMAND_SET_BC         0x0A // [Brightness Value]
#define CUBE_COMMAND_CLEAR          0x0B // (no data)
#define CUBE_COMMAND_SET_PIXEL      0x10 // [Layer, Row, Column, R, G, B]
#define CUBE_COMMAND_SET_ALL        0x11 // [R1, G1, B1, R2, ...]
#define CUBE_COMMAND_START_TEXT     0x20 // [R, G, B, Char1, Char2, Char3, ...]
#define CUBE_COMMAND_STOP_TEXT      0x21 // (no data)

// Ethernet Op-Codes
#define CUBE_ETH_RESET              0x01
#define CUBE_EHT_IDLE               0x02
#define CUBE_ETH_CLEAR              0x0A
#define CUBE_ETH_DCS                0x0B
#define CUBE_ETH_ROTATE             0x0C
#define CUBE_ETH_ROTATE_LAYER       0x0D
#define CUBE_ETH_WRITE_ALL          0x10
#define CUBE_ETH_WRITE_PIXEL        0x11
#define CUBE_ETH_WRITE_CHANNEL      0x12
#define CUBE_ETH_WRITE_TEXT_SCROLL  0x20
#define CUBE_ETH_WRITE_TEXT_STATIC  0x21
#define CUBE_EHT_WRITE_TEXT_INSERT  0x22
#define CUBE_ETH_WATERFALL          0x30
#define CUBE_ETH_SPHERE             0x31

typedef enum {
    IDLE,
    READ_LENGTH_MSB,
    READ_LENGTH_LSB,
    READ_COMMAND,
    READ_DATA,
    READ_CHECKSUM
} PROCESS_STATE;

typedef struct {
    // Variables for base cube
    uint8_t GCS[CUBE_LAYER_COUNT][GCS_LAYER_SIZE];
    uint8_t GCS_OVERLAY[CUBE_LAYER_COUNT][GCS_LAYER_SIZE];
    uint8_t GCS_WRITE[CUBE_LAYER_COUNT][GCS_LAYER_SIZE];
    uint8_t current_layer;
    uint8_t rotation_counter;

    // Variables for the scrolling text
    uint8_t string[CUBE_STRING_MAX_LENGTH];
    uint8_t string_length;
    uint8_t string_index;
    uint8_t string_line;
    uint16_t string_R, string_G, string_B;

    // Variables for input frame data
    PROCESS_STATE frame_state;
    uint8_t frame_buffer[CUBE_FRAME_BUFFER_SIZE];
    uint8_t frame_checksum;
    uint32_t frame_length;
    uint32_t frame_index;
    uint32_t frame_command;
    uint32_t frame_escape;
} CUBE_DATA;

void Cube_Init(CUBE_DATA *data, uint8_t BC);
void Cube_Timer_Interrupt(void);

// Callbacks on completion of DCS/GCS writes
void Cube_DCS_Write_Callback(void);
void Cube_GCS_Write_Callback(void);

// Cube control functions
void Cube_Write_DCS(uint8_t BC);
void Cube_Clear(void);
void Cube_Set_All(uint16_t R, uint16_t G, uint16_t B);
void Cube_Set_Layer(uint8_t layer, uint16_t R, uint16_t G, uint16_t B);
void Cube_Set_Row(uint8_t row, uint16_t R, uint16_t G, uint16_t B);
void Cube_Set_Column(uint8_t column, uint16_t R, uint16_t G, uint16_t B);
void Cube_Set_Pixel(uint8_t layer, uint8_t row, uint8_t column, uint16_t R, uint16_t G, uint16_t B);
void Cube_Get_Pixel(uint8_t layer, uint8_t row, uint8_t column, uint16_t* R, uint16_t* G, uint16_t* B);
void Cube_Move_Pixel(uint8_t layer1, uint8_t row1, uint8_t column1, uint8_t layer2, uint8_t row2, uint8_t column2);
void Cube_Set_Sphere(uint8_t layer, uint8_t R, uint8_t G, uint8_t B);
void Cube_Set_Shell(uint8_t layer, uint8_t R, uint8_t G, uint8_t B);
void Cube_Rotate_Shell(uint8_t shell, uint8_t direction);
void Cube_Rotate(uint8_t direction);
void Cube_Shift_Row(uint8_t direction);

void Cube_Shift_Waterfall(uint8_t *values);
void Cube_Shift_Waterfall2(uint8_t *values);

// Overlay control functions
void Cube_Overlay_Clear(void);
void Cube_Overlay_Set_Pixel(uint8_t layer, uint8_t row, uint8_t column, uint16_t R, uint16_t G, uint16_t B);
void Cube_Overlay_Get_Pixel(uint8_t layer, uint8_t row, uint8_t column, uint16_t* R, uint16_t* G, uint16_t* B);
void Cube_Overlay_Move_Pixel(uint8_t layer1, uint8_t row1, uint8_t column1, uint8_t layer2, uint8_t row2, uint8_t column2);
void Cube_Overlay_Rotate_Shell(uint8_t shell, uint8_t direction);

// Text control functions
void Cube_Text_Init(uint8_t *string, uint8_t length, uint16_t R, uint16_t G, uint16_t B);
void Cube_Text_Update(void);
void Cube_Text_Insert(uint8_t c, uint16_t R, uint16_t G, uint16_t B, uint16_t delay);
void Cube_Text_Single_Char_Interupt(void);
void Cube_Text_Interrupt(void);

// Data stream in control functions
// UART functions
void Cube_Data_In(uint8_t c);
void Cube_Data_In_Process_Frame(void);
void Cube_Data_Direct_Write_All(uint8_t *buffer);
// Ethernet functions
void Cube_Ethernet_Frame_In(void);

#endif	/* CUBE_H */

