#include "defines.h"
#include "CONTROLLERS.h"
#include "I2C1.h"

static CONTROLLER_DATA *ctrl_data_p;

void Controller_Init(CONTROLLER_DATA *data,
        void (*change_callback)(uint8_t controller, CTRL_BTN_STATUS values)) {

    if (data != NULL)
        ctrl_data_p = data;
    ctrl_data_p->change_callback = change_callback;

    // Variable initialization
    int i, j;
    for (i = 0; i < CONTROLLER_MAX_COUNT; i++) {
        ctrl_data_p->connected_controllers[i] = 0x0;
        ctrl_data_p->led_status[i].w[i] = 0x0;
        ctrl_data_p->btn_prev[i].w = 0x0;
        ctrl_data_p->btn_curr[i].w = 0x0;
        for (j = 0; j < 16; j++) {
            ctrl_data_p->led_status[i].w[j] = 0x0;
        }
    }

    ctrl_data_p->connected_count = 0x0;
    ctrl_data_p->next_poll = 0x0;

    // Poll to see which controllers are connected
    Controller_Poll_Connected();
}

void Controller_Poll_Connected(void) {
    uint8_t buffer[2];
    uint8_t result, i;
    uint8_t address = CONTROLLER_PREFIX_ADDRESS + CONTROLLER_START_ADDRESS;

    // Attempt to contact each controller to see if its connected
    for (i = 0; i < CONTROLLER_MAX_COUNT; i++) {
        I2C1_Master_Restart(address + i, CONTROLLER_CMD_READ, 1);
        do {
            result = I2C1_Get_Status();
        } while (!result);
        if (result == I2C1_RECV_OK) {
            uint8_t length = I2C1_Read_Buffer(buffer);
            // If a controller is connected, save its address
            ctrl_data_p->connected_controllers[ctrl_data_p->connected_count] = address + i;
            ctrl_data_p->connected_count++;
        }

        // A small delay is needed between polls
        Delay_US(500);
    }

    // Show the number of controllers connected
    if (ctrl_data_p->connected_count & 0x1)
        LED1_LAT = 1;
    if (ctrl_data_p->connected_count & 0x2)
        LED2_LAT = 1;
    if (ctrl_data_p->connected_count & 0x4)
        LED3_LAT = 1;
    if (ctrl_data_p->connected_count & 0x8)
        LED4_LAT = 1;
}

void Controller_Update(void) {
    uint8_t buffer[2];
    uint8_t result, length;
    CTRL_BTN_STATUS btn_change;

    uint8_t controller = ctrl_data_p->next_poll;

    // Store the last read values
    ctrl_data_p->btn_prev[controller] = ctrl_data_p->btn_curr[controller];

    // Read in the button values for each controller
    I2C1_Master_Restart(ctrl_data_p->connected_controllers[controller], CONTROLLER_CMD_READ, 1);
    do {
        result = I2C1_Get_Status();
    } while (!result);
    // Save the read values
    length = I2C1_Read_Buffer(buffer);
    ctrl_data_p->btn_curr[controller].w = buffer[0];

    // Determine if a change was made between the current and previous values
    if (ctrl_data_p->btn_curr[controller].w != ctrl_data_p->btn_prev[controller].w) {
        // Determine if a button was pressed (unpressed->pressed)
        btn_change.w = ctrl_data_p->btn_prev[controller].w ^ ctrl_data_p->btn_curr[controller].w;
        btn_change.w &= ctrl_data_p->btn_curr[controller].w;

        // If a button was pressed, call the saved callback
        if (btn_change.w) {
            if (ctrl_data_p->change_callback != NULL) {
                ctrl_data_p->change_callback(controller, btn_change);
            }
        }
    }

    ctrl_data_p->next_poll = (controller == ctrl_data_p->connected_count - 1) ? 0x0 : controller+1;
}

void Controller_Set_Left_Leds(uint8_t controller, uint8_t value) {
    uint8_t result, i;
    uint8_t buffer[18];

    for (i = 0; i < 4; i++) {
        if (value & (0x01 << i))
            ctrl_data_p->led_status[controller].w[i+12] = CONTROLLER_BRIGHTNESS_HIGH;
        else
            ctrl_data_p->led_status[controller].w[i+12] = 0x00;
    }

    // Write the LED value to the controller
    buffer[0] = CONTROLLER_CMD_WRITE;
    for (i = 0; i < 16; i++)
        buffer[i+1] = ctrl_data_p->led_status[controller].w[i];
    I2C1_Master_Send(ctrl_data_p->connected_controllers[controller], buffer, 17);
    do {
        result = I2C1_Get_Status();
    } while (!result);

    Delay_MS(1);
}

void Controller_Set_Middle_Leds(uint8_t controller, uint8_t value) {
    uint8_t result, i;
    uint8_t buffer[18];

    for (i = 0; i < 8; i++) {
        if (value & (0x01 << i))
            ctrl_data_p->led_status[controller].w[i] = CONTROLLER_BRIGHTNESS_HIGH;
        else
            ctrl_data_p->led_status[controller].w[i] = 0x00;
    }

    // Write the LED value to the controller
    buffer[0] = CONTROLLER_CMD_WRITE;
    for (i = 0; i < 16; i++)
        buffer[i+1] = ctrl_data_p->led_status[controller].w[i];
    I2C1_Master_Send(ctrl_data_p->connected_controllers[controller], buffer, 17);
    do {
        result = I2C1_Get_Status();
    } while (!result);

    Delay_MS(1);
}

void Controller_Set_Right_Leds(uint8_t controller, uint8_t value) {
    uint8_t result, i;
    uint8_t buffer[18];

    for (i = 0; i < 4; i++) {
        if (value & (0x01 << i))
            ctrl_data_p->led_status[controller].w[i+8] = CONTROLLER_BRIGHTNESS_HIGH;
        else
            ctrl_data_p->led_status[controller].w[i+8] = 0x00;
    }

    // Write the LED value to the controller
    buffer[0] = CONTROLLER_CMD_WRITE;
    for (i = 0; i < 16; i++)
        buffer[i+1] = ctrl_data_p->led_status[controller].w[i];
    I2C1_Master_Send(ctrl_data_p->connected_controllers[controller], buffer, 17);
    do {
        result = I2C1_Get_Status();
    } while (!result);

    Delay_MS(1);
}

uint8_t Controller_Get_Connected(void) {
    return ctrl_data_p->connected_count;
}

void Controller_Set_Active(uint8_t controller) {
    uint8_t buffer[2];
    uint8_t result;

    buffer[0] = CONTROLLER_CMD_ACTIVE;
    I2C1_Master_Send(ctrl_data_p->connected_controllers[controller], buffer, 1);
    do {
        result = I2C1_Get_Status();
    } while (!result);

    Delay_MS(1);
}

void Controller_Set_Idle(uint8_t controller) {
    uint8_t buffer[2];
    uint8_t result;

    buffer[0] = CONTROLLER_CMD_RESET;
    I2C1_Master_Send(ctrl_data_p->connected_controllers[controller], buffer, 1);
    do {
        result = I2C1_Get_Status();
    } while (!result);

    Delay_MS(1);
}