#ifndef ETHERNET_H
#define	ETHERNET_H

#define ETH_TX_DESCRIPTOR_COUNT 2
#define ETH_RX_DESCRIPTOR_COUNT 2
#define ETH_TX_ED_BUFFER_SIZE   2032
#define ETH_RX_ED_BUFFER_SIZE   2032

#ifdef CEREBOT_MX7CK
#define PHY_RESET_TRIS  TRISAbits.TRISA6
#define PHY_RESET_LAT   LATAbits.LATA6
#endif
#ifdef CEREBOT_32MX7
#define PHY_RESET_TRIS  TRISEbits.TRISE9
#define PHY_RESET_LAT   LATEbits.LATE9
#endif
#define ETH_MDC_TRIS    TRISDbits.TRISD11
#define ETH_MDIO_TRIS   TRISDbits.TRISD8
#define ETH_TXEN_TRIS   TRISDbits.TRISD6
#define ETH_TXD0_TRIS   TRISFbits.TRISF1
#define ETH_TXD1_TRIS   TRISFbits.TRISF0
#define ETH_RXCLK_TRIS  TRISGbits.TRISG9
#define ETH_RXDV_TRIS   TRISGbits.TRISG8
#define ETH_RXD0_TRIS   TRISBbits.TRISB12
#define ETH_RXD1_TRIS   TRISBbits.TRISB13
#define ETH_RXERR_TRIS  TRISBbits.TRISB11

#define PHY_ADDRESS     0x0

typedef union {
    struct {
        uint8_t BYTE_0;
        uint8_t BYTE_1;
        uint8_t BYTE_2;
        uint8_t BYTE_3;
        uint8_t BYTE_4;
        uint8_t BYTE_5;
    };
    uint8_t bytes[6];
} ETH_MAC_ADDRESS;

typedef union {
    struct {
        // Bits 31:0
        unsigned BYTE_COUNT             : 16;   // Total bytes in frame not counting collided bytes
        unsigned COLLISION_COUNT        : 4;    // Number of collisions current packet incurred durrent transmit attempts
        unsigned CRC_ERROR              : 1;    // Attached CRC did not match the internal generated CRC
        unsigned LENGTH_CHECK_ERROR     : 1;    // Frame length field value in packet does not match actual data byte length and is not a Type field
        unsigned LENGTH_OUT_OF_RANGE    : 1;    // Frame type/length field was larger than 1500 bytes
        unsigned DONE                   : 1;    // Transmit of packet was completed
        unsigned MULTICASE              : 1;    // Destination address was a multicast address
        unsigned BROADCAST              : 1;    // Destination address was a broadcast address
        unsigned PACKET_DEFER           : 1;    // Packet was deferred for at least one attempt
        unsigned EXCESSIVE_DEFER        : 1;    // Packet was defered in excess of 6071/24287 nibble(100Mbps)/bit(10Mbps) times
        unsigned MAXIMUM_COLLISION      : 1;    // Packet aborted, number of collisions exceeded RETX
        unsigned LATE_COLLISION         : 1;    // Collision occurred beyond the collision window (512 bit times)
        unsigned GIANT                  : 1;    // Frame byte count greater than MACMAXF
        unsigned UNDERRUN               : 1;    // Failed to transfer complete packet to the transmit MAC module
        // Bits 63:32
        unsigned BYTES_TRANSMITTED      : 16;   // Total bytes transmitted on wire (including collisions)
        unsigned CONTROL_FRAME          : 1;    // Frame transmitted was a control frame
        unsigned PAUSE_CONTROL_FRAME    : 1;    // Frame transmitted was a control frame with a valid PAUSE Op code
        unsigned BACKPRESSURE           : 1;    // Carrier-sense method backpressure was previously applied
        unsigned VLAN_TAGGED            : 1;    // Frame length/type field contained 0x8100 (VLAN protocol identifier)
        unsigned                        : 12;
    };
    uint32_t registers[2];
} ETH_TRANSMIT_STATUS_VECTOR;

typedef union {
    struct {
        // Bits 31:0
        unsigned BYTE_COUNT             : 16;   // Length of received frame
        unsigned LONG_DROP_EVENT        : 1;    // Packet over 50000 bit times occured or packet since last RSV was dropped
        unsigned RXDV_EVENT             : 1;    // Last receive event seen not long enough to be a valid packet
        unsigned CARRIER_EVENT          : 1;    // Carrier event detected, noted, and reported
        unsigned CODE_VIOLATION         : 1;    // MII data does not represent a valid data code when MRXER asserted
        unsigned CRC_ERROR              : 1;    // Frame CRC does not match the CRC calculated by the receiver MAC
        unsigned LENGTH_CHECK_ERROR     : 1;    // Frame length field value doe snot match the actual data byte length
        unsigned LENGTH_OUT_OF_RANGE    : 1;    // Frame type/length field was larger than 1500 bytes
        unsigned RECEIVE_OK             : 1;    // Packet has a valid CRC and no symbol errors
        unsigned MULTICAST              : 1;    // Packet had a valid multicast address
        unsigned BROADCAST              : 1;    // Packet had a valid broadcast address
        unsigned DRIBBLE_NIBBLE         : 1;    // An additional 1-7 bits received after packet
        unsigned CONTROL_FRAME          : 1;    // Frame recognized as a control frame
        unsigned PAUSE_CONTROL_FRAME    : 1;    // Frame recognized as a control frame with a valid PAUSE Op code
        unsigned UNKNOWN_OP_CODE        : 1;    // Frame recognized as a control frame but with an unknown Op code
        unsigned VLAN_TAGGED            : 1;    // Frame recognized as a VLAN tagged frame
        unsigned                        : 1;
        // Bits 63:32;
        unsigned PKT_CHECKSUM           : 16;   // RX packet payload checksum of this descriptor's packet
        unsigned                        : 8;
        unsigned RUNT_PACKET            : 1;    // Runt packet
        unsigned BROADCAST_OR_NOT_DEST  : 1;    // NOT unicast match AND NOT multicast match
        unsigned HASH_TABLE_MATCH       : 1;    // Hash table match
        unsigned MAGIC_PACKET_MATCH     : 1;    // Magic packet match
        unsigned PATTERN_MATCH          : 1;    // Pattern match
        unsigned UNICAST_MATCH          : 1;    // Unicast match
        unsigned BROADCAST_MATCH        : 1;    // Broadcast match
        unsigned MULTICAST_MATCH        : 1;    // Multicast match
    };
    uint32_t registers[2];
} ETH_RECEIVE_STATUS_VECTOR;

typedef struct {
    unsigned        : 7;
    unsigned EOWN   : 1;            // Ethernet controller own bit (1 = ED owned by controller, do not modify)
    unsigned NPV    : 1;            // Next ED pointer valid enable bit (1 = NEXT_ED field exists)
    unsigned        : 7;
    unsigned BYTE_COUNT : 11;       // Number of bytes to be transmited for this descriptor (1-2047)
    unsigned        : 3;
    unsigned EOP    : 1;            // End of packet enable bit (1 = transmit end of packet delimiter)
    unsigned SOP    : 1;            // Start of packet enable bit (1 = transmit start of packet delimiter)
    uint32_t BUFFER_ADDR;       // Starting point address of the data buffer
    ETH_TRANSMIT_STATUS_VECTOR TSV; // Transmit status vector bits
} ETH_TX_ETHERNET_DESCRIPTOR;

typedef struct {
    unsigned        : 7;
    unsigned EOWN   : 1;            // Ethernet controller own bit (1 = ED owned by controller, do not modify)
    unsigned NPV    : 1;            // Next ED pointer valid enable bit (1 = NEXT_ED field exists)
    unsigned        : 7;
    unsigned BYTE_COUNT : 11;       // Number of bytes to be transmited for this descriptor (1-2047)
    unsigned        : 3;
    unsigned EOP    : 1;            // End of packet enable bit (1 = transmit end of packet delimiter)
    unsigned SOP    : 1;            // Start of packet enable bit (1 = transmit start of packet delimiter)
    uint32_t BUFFER_ADDR;       // Starting point address of the data buffer
    ETH_RECEIVE_STATUS_VECTOR RSV;  // Transmit status vector bits
} ETH_RX_ETHERNET_DESCRIPTOR;

typedef struct {
    ETH_TX_ETHERNET_DESCRIPTOR descriptor[ETH_TX_DESCRIPTOR_COUNT];
    uint32_t next_ED;
} ETH_TX_DESCRIPTOR_TABLE;

typedef struct {
    ETH_RX_ETHERNET_DESCRIPTOR descriptor[ETH_RX_DESCRIPTOR_COUNT];
    uint32_t next_ED;
} ETH_RX_DESCRIPTOR_TABLE;

typedef struct {
    ETH_TX_DESCRIPTOR_TABLE TX_ED_table;
    ETH_RX_DESCRIPTOR_TABLE RX_ED_table;
    uint8_t TX_ED_buffer[ETH_TX_DESCRIPTOR_COUNT][ETH_TX_ED_BUFFER_SIZE];
    uint8_t RX_ED_buffer[ETH_RX_DESCRIPTOR_COUNT][ETH_RX_ED_BUFFER_SIZE];
    uint8_t TX_descriptor_index;
    uint8_t RX_descriptor_index;
    void (*tx_callback)(void);
    void (*rx_callback)(void);
} ETH_DATA;

void ETH_Init(ETH_DATA *data, void(*tx_callback)(void), void(*rx_callback)(void));

uint16_t ETH_PHY_Read(uint8_t address, uint8_t reg);
void ETH_PHY_Write(uint8_t address, uint8_t reg, uint16_t value);

uint8_t ETH_Recv_Queue(void);
uint8_t ETH_Read_Packet(uint8_t *buffer, uint16_t *length);
uint8_t ETH_Write_Packet(ETH_MAC_ADDRESS dest, ETH_MAC_ADDRESS src, uint16_t length, uint8_t *buffer);

#endif	/* ETHERNET_H */

