#include "MainWindow.h"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent)
{
    // Set central widget
    centralWidget = new QWidget();
    setCentralWidget(centralWidget);

    // Misc button group
    groupOther = new QGroupBox("Other");
    btnMacro = new QPushButton("&Macros");
    btnPaste = new QPushButton("&Paste");
    ToggleGroupLayout(false);

    // Initialize serial widget
    serialWidget = new SerialWidget();
    groupSerialInit = new QGroupBox("Serial Connection");
    QGridLayout *serialInitLayout = new QGridLayout();
    serialInitLayout->setContentsMargins(0, 0, 0, 0);
    serialInitLayout->addWidget(serialWidget);
    groupSerialInit->setLayout(serialInitLayout);
    connect(serialWidget, SIGNAL(UpdateStatus(QString)), this, SLOT(UpdateSerialStatus(QString)));

    // Initialize data widget
    ioWidget = new IOWidget();
    groupSerialData = new QGroupBox("Data");
    QGridLayout *serialDataLayout = new QGridLayout();
    serialDataLayout->setContentsMargins(0, 0, 0, 0);
    serialDataLayout->addWidget(ioWidget);
    groupSerialData->setLayout(serialDataLayout);
    connect(serialWidget, SIGNAL(ReceivedByte(QByteArray)), ioWidget, SLOT(ProcessReceivedByte(QByteArray)));
    connect(serialWidget, SIGNAL(Connected(bool)), ioWidget, SLOT(EnableTransmit(bool)));
    connect(ioWidget, SIGNAL(TransmitByteArray(QByteArray)), serialWidget, SIGNAL(TransmitByteArray(QByteArray)));

    // Initialize macro widget
    macroDockWidget = new QDockWidget("Macro Controller", this);
    macroWidget = new MacroWidget(macroDockWidget);
    macroDockWidget->setWidget(macroWidget);
    macroDockWidget->setAllowedAreas(Qt::LeftDockWidgetArea | Qt::RightDockWidgetArea);
    macroDockWidget->hide();
    addDockWidget(Qt::RightDockWidgetArea, macroDockWidget);
    connect(macroWidget, SIGNAL(TransmitText(QByteArray)), ioWidget, SLOT(PreprocessTransmit(QByteArray)));
    connect(serialWidget, SIGNAL(Connected(bool)), macroWidget, SLOT(EnableTransmit(bool)));

    // Initialize paste controller widget
    pasteDockWidget = new QDockWidget("Paste Controller", this);
    pasteWidget = new PasteController(pasteDockWidget);
    pasteDockWidget->setWidget(pasteWidget);
    pasteDockWidget->setFloating(true);
    pasteDockWidget->hide();
    connect(serialWidget, SIGNAL(Connected(bool)), pasteWidget, SLOT(EnableTransmit(bool)));
    connect(pasteWidget, SIGNAL(TransmitData(QByteArray)), ioWidget, SLOT(PreprocessTransmit(QByteArray)));
    connect(serialWidget, SIGNAL(ReceivedByte(QByteArray)), pasteWidget, SLOT(ProcessData(QByteArray)));

    // Connect local widgets
    connect(btnMacro, SIGNAL(clicked()), macroDockWidget->toggleViewAction(), SLOT(trigger()));
    connect(btnPaste, SIGNAL(clicked()), pasteDockWidget->toggleViewAction(), SLOT(trigger()));

    // Main layout
    QGridLayout *mainLayout = new QGridLayout();
    mainLayout->addWidget(groupSerialInit, 0, 0, 1, 1, Qt::AlignLeft);
    mainLayout->addWidget(groupOther, 0, 1, 1, 1, Qt::AlignLeft);
    mainLayout->addWidget(groupSerialData, 1, 0, 1, 2);
    mainLayout->setColumnStretch(0, 0);
    mainLayout->setColumnStretch(1, 1);
    centralWidget->setLayout(mainLayout);
    connect(serialWidget, SIGNAL(Connected(bool)), this, SLOT(ToggleGroupLayout(bool)));

    setWindowTitle("Marlin Controller");
    setWindowIcon(QIcon(":/External/Resources/Icon_16.bmp"));

    labelSerialStatus = new QLabel("Disconnected");
    statusBar()->addPermanentWidget(labelSerialStatus);
}

MainWindow::~MainWindow()
{

}

void MainWindow::UpdateStatus(QString string)
{
    statusBar()->showMessage(string, STATUS_TIMEOUT_MS);
}

void MainWindow::UpdateSerialStatus(QString string)
{
    labelSerialStatus->setText(string);
}

void MainWindow::ToggleGroupLayout(bool toggle)
{
    QGridLayout *layout = qobject_cast<QGridLayout*>(groupOther->layout());
    if (layout == NULL) layout = new QGridLayout();
    layout->removeWidget(btnMacro);
    layout->removeWidget(btnPaste);
    if (toggle) {
        layout->addWidget(btnMacro, 0, 0);
        layout->addWidget(btnPaste, 0, 1);
    } else {
        layout->addWidget(btnMacro, 0, 0);
        layout->addWidget(btnPaste, 1, 0);
    }
    groupOther->setLayout(layout);
    groupOther->repaint();
}
