/**************************************************************************
 * Copyright 2011 Jules White
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/
package ioio.debugger;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.jboss.netty.channel.Channel;
import org.jboss.netty.channel.ChannelEvent;
import org.jboss.netty.channel.ChannelFuture;
import org.jboss.netty.channel.ChannelFutureListener;
import org.jboss.netty.channel.ChannelHandlerContext;
import org.jboss.netty.channel.ChannelStateEvent;
import org.jboss.netty.channel.ExceptionEvent;
import org.jboss.netty.channel.MessageEvent;
import org.jboss.netty.channel.SimpleChannelUpstreamHandler;

import android.util.Log;

/**
 * Handles a client-side channel.
 * 
 * @author <a href="http://www.jboss.org/netty/">The Netty Project</a>
 * @author <a href="http://gleamynode.net/">Trustin Lee</a>
 * 
 * @version $Rev: 2121 $, $Date: 2010-02-02 09:38:07 +0900 (Tue, 02 Feb 2010) $
 */
public class NetworkClientHandler<T> extends SimpleChannelUpstreamHandler {

	private static final String TAG = NetworkClientHandler.class.getName();
	
	private NetworkMessageHandler<T> listener_;

	private Channel channel_;

	private static final Logger logger = Logger
			.getLogger(NetworkClientHandler.class.getName());

	public void disconnect(){
		channel_.disconnect();
	}
	
	@Override
	public void handleUpstream(ChannelHandlerContext ctx, ChannelEvent e)
			throws Exception {
		if (e instanceof ChannelStateEvent) {
			logger.info(e.toString());
		}
		super.handleUpstream(ctx, e);
	}

	// @Override
	public void channelConnected(ChannelHandlerContext ctx, ChannelStateEvent e)
			throws Exception {

		channel_ = e.getChannel();

		channel_.getCloseFuture().addListener(new ChannelFutureListener() {

			@Override
			public void operationComplete(ChannelFuture future)
					throws Exception {
				listener_.channelClosed(channel_);
			}
		});
	}

	@Override
	public void messageReceived(ChannelHandlerContext ctx, MessageEvent e) {
		if (listener_ != null) {
			@SuppressWarnings("unchecked")
			T msg = (T) e.getMessage();
			listener_.received(msg);
		} else {
			System.err.println(e.getMessage());
		}
	}

	public void send(NetworkIOIOProtocol.TrackingMessage msg) {
		channel_.write(msg);
	}

	@Override
	public void exceptionCaught(ChannelHandlerContext ctx, ExceptionEvent e) {
		logger.log(Level.WARNING, "Unexpected exception from downstream.",
				e.getCause());
		e.getChannel().close();

		if(listener_ != null)
			listener_.channelException(e.getCause());
		else
			Log.e(TAG, "Error communicating with server",e.getCause());
	}

	public NetworkMessageHandler<T> getListener() {
		return listener_;
	}

	public void setListener(NetworkMessageHandler<T> listener) {
		listener_ = listener;
	}

}
