package ioio.debugger;

import java.util.List;

import ioio.debugger.NetworkIOIOProtocol.Message;
import ioio.debugger.NetworkIOIOProtocol.Message.Builder;
import ioio.debugger.NetworkIOIOProtocol.TrackingMessage;

import org.jboss.netty.channel.Channel;

import android.os.Handler;

public class NetworkClientConnectionHandler implements NetworkMessageHandler<TrackingMessage> {

	// Message handler for sending messages to server
	private NetworkClientHandler<NetworkIOIOProtocol.TrackingMessage> msgHandler_;
		
	private Boolean connected_ = false;
	private boolean connectionOk_ = false;
	private long reconnectInterval_ = 3000;
	private boolean autoReconnect_ = false;
	private String server_;
	private int port_;
	private IOIODebuggerActivity parent_;
	private Thread connection_;
	private Handler handler_ = new Handler();
	
	public NetworkClientConnectionHandler(IOIODebuggerActivity parent) {
		parent_ = parent;
		connection_ = new Thread();
	}
	
	public void setConnectionSettings(final String server, final int port, final boolean autoReconnect, final long reconnectInterval) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				server_ = server;
				port_ = port;
				autoReconnect_ = autoReconnect;
				reconnectInterval_ = reconnectInterval;
			}
		});
	}
	
	@Override
	public void channelException(Throwable e) {
		synchronized (connected_) {
			parent_.updateServerStatus("Exception occured contacting server");
			connected_ = false;
			connectionOk_ = false;
		}
	}

	@Override
	public void channelClosed(Channel c) {
		synchronized (connected_) {
			parent_.updateServerStatus("Connection to server closed");
			connected_ = false;
			connectionOk_ = false;
		}
	}

	@Override
	public void received(final TrackingMessage msg) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				List<Message> messageList;
				switch(msg.getType()) {
				case 11:
					parent_.updateServerStatus("Connected to server  [ " + server_ + " :: " + port_ + " ]");
					connectionOk_ = true;
					parent_.sendAllData();
					break;
				case 12:
					parent_.updateServerStatus("Connection to server rejected. Try again?");
					disconnectFromServer();
					break;
				case 13:
					parent_.hardReset();
					break;
				case 50:
					messageList = msg.getMessageList();
					for (Message message : messageList) {
						String type = message.getType();
						if (type.compareTo("InAnalog") == 0) {
							CreateWidget.addIA(message.getPin(), message.getFreq());
							parent_.processAddWidget(IOIODebuggerActivity.ACTIVITY_INPUT_ANALOG);
						} else if (type.compareTo("InDigital") == 0) {
							CreateWidget.addID(message.getPin(), message.getFreq(), message.getMode());
							parent_.processAddWidget(IOIODebuggerActivity.ACTIVITY_INPUT_DIGITAL);
						} else if (type.compareTo("OutDigital") == 0) {
							CreateWidget.addOD(message.getPin(), message.getMode(), message.getBoolean());
							parent_.processAddWidget(IOIODebuggerActivity.ACTIVITY_OUTPUT_DIGITAL);
						}
					}
					break;
				case 51:
					messageList = msg.getMessageList();
					for (Message message : messageList) {
						parent_.stopViewWidget(message.getPin());
					}
					break;
				case 52:
					messageList = msg.getMessageList();
					for (Message message : messageList) {
						parent_.removeViewWidget(message.getPin());
					}
				}
			}
		});
	}

	@SuppressWarnings("unchecked")
	public boolean connectToServer() {

		try {
			synchronized (connected_) {
				if (connected_)
					return true;

				parent_.updateServerStatus("Connecting to server...");

				msgHandler_ = NetworkClient.connect(server_, port_);
				msgHandler_.setListener(this);
				connected_ = true;
				
				// Wait a bit for the connection to finish initialization
				Thread.sleep(500);
				
				// Queries if connection is ok
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(10)
						.build();
				
				msgHandler_.send(newMsg);
			}
			return true;
		} catch (Exception e) {
			parent_.updateServerStatus("Unable to connect to server");
			return false;
		}

	}
	
	private class ConnectWorker implements Runnable {

		public void run() {
			doConnect();
		}

		public void doConnect() {
			try {
				if (!connectToServer()) {
					while (autoReconnect_ && !connectToServer()) {
						parent_.updateServerStatus("Waiting to reconnect to server...");
						Thread.sleep(reconnectInterval_);
					}
				}
			} catch (Exception e) {
				doConnect();
			}
		}
	}
	
	public void startConnect() {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				ConnectWorker w = new ConnectWorker();
				connection_ = new Thread(w);
				connection_.start();
			}
		});
	}

	public void stopAutoReconnect() {
		autoReconnect_ = false;
	}
	
	public void disconnectFromServer() {
		synchronized (connected_) {
			connected_ = false;
			connectionOk_ = false;
			autoReconnect_ = false;
			msgHandler_.disconnect();
			parent_.updateServerStatus("Disconnected from server");
		}
	}
	
	public Boolean isConnected() {
		return connected_;
	}
	
	public boolean connectedOk() {
		return connectionOk_;
	}
	
	
	public void notifyACK_Success() {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(1)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void notifyACK_Failure() {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(2)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void notifyStartIA(final int pin, final long freq) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				Builder message = Message.newBuilder()
						.setPin(pin)
						.setFreq(freq)
						.setType("InAnalog");
				
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(50)
						.addMessage(message)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void notifyStartID(final int pin, final long freq, final String mode) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				Builder message = Message.newBuilder()
						.setPin(pin)
						.setFreq(freq)
						.setMode(mode)
						.setType("InDigital");
				
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(50)
						.addMessage(message)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void notifyStartOD(final int pin, final String state, final boolean openDrain) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				Builder message = Message.newBuilder()
						.setPin(pin)
						.setState(state)
						.setBoolean(openDrain)
						.setType("OutDigital");
				
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(50)
						.addMessage(message)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void notifyStopPin(final int pin) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				Builder message = Message.newBuilder()
						.setPin(pin);
				
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(51)
						.addMessage(message)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void notifyRemovePin(final int pin) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				Builder message = Message.newBuilder()
						.setPin(pin);
				
				TrackingMessage newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(52)
						.addMessage(message)
						.build();
				
				msgHandler_.send(newMsg);
			}
		});
	}
	
	public void sendData(final int pin, final Double[] data) {
		handler_.post(new Runnable() {
			
			@Override
			public void run() {
				ioio.debugger.NetworkIOIOProtocol.TrackingMessage.Builder newMsg = TrackingMessage.newBuilder()
						.setId(0)
						.setType(70);
				
				Builder message = Message.newBuilder().setPin(pin);
				for (double d : data) {
					message.addData(d);
				}
				
				newMsg.addMessage(message);
				msgHandler_.send(newMsg.build());
			}
		});
	}
}
